@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.security

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.security.events.SecurityEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [SecurityDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * A dummy response object for the [SecurityDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * Request object containing input parameters for the [SecurityDomain.setIgnoreCertificateErrors]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetIgnoreCertificateErrorsRequest(
  /**
   * If true, all certificate errors will be ignored.
   */
  public val ignore: Boolean,
)

/**
 * A dummy response object for the [SecurityDomain.setIgnoreCertificateErrors] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetIgnoreCertificateErrorsResponse

/**
 * Request object containing input parameters for the [SecurityDomain.handleCertificateError]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class HandleCertificateErrorRequest(
  /**
   * The ID of the event.
   */
  public val eventId: Int,
  /**
   * The action to take on the certificate error.
   */
  public val action: CertificateErrorAction,
)

/**
 * A dummy response object for the [SecurityDomain.handleCertificateError] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object HandleCertificateErrorResponse

/**
 * Request object containing input parameters for the [SecurityDomain.setOverrideCertificateErrors]
 * command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class SetOverrideCertificateErrorsRequest(
  /**
   * If true, certificate errors will be overridden.
   */
  public val `override`: Boolean,
)

/**
 * A dummy response object for the [SecurityDomain.setOverrideCertificateErrors] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object SetOverrideCertificateErrorsResponse

/**
 * Security
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security)
 */
public class SecurityDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out SecurityEvent>> =
      mapOf(
      "Security.certificateError" to serializer<SecurityEvent.CertificateError>(),
      "Security.visibleSecurityStateChanged" to
          serializer<SecurityEvent.VisibleSecurityStateChanged>(),
      "Security.securityStateChanged" to serializer<SecurityEvent.SecurityStateChanged>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<SecurityEvent> = session.typedEvents(deserializersByEventName)

  /**
   * There is a certificate error. If overriding certificate errors is enabled, then it should be
   * handled with the `handleCertificateError` command. Note: this event does not fire if the
   * certificate error has been allowed internally. Only one client per target should override
   * certificate errors at the same time.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#event-certificateError)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public fun certificateErrorEvents(): Flow<SecurityEvent.CertificateError> =
      session.typedEvents("Security.certificateError")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("certificateErrorEvents()"),
  )
  public fun certificateError(): Flow<SecurityEvent.CertificateError> = certificateErrorEvents()

  /**
   * The security state of the page changed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#event-visibleSecurityStateChanged)
   */
  @ExperimentalChromeApi
  public fun visibleSecurityStateChangedEvents(): Flow<SecurityEvent.VisibleSecurityStateChanged> =
      session.typedEvents("Security.visibleSecurityStateChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("visibleSecurityStateChangedEvents()"),
  )
  public fun visibleSecurityStateChanged(): Flow<SecurityEvent.VisibleSecurityStateChanged> =
      visibleSecurityStateChangedEvents()

  /**
   * The security state of the page changed. No longer being sent.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#event-securityStateChanged)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public fun securityStateChangedEvents(): Flow<SecurityEvent.SecurityStateChanged> =
      session.typedEvents("Security.securityStateChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("securityStateChangedEvents()"),
  )
  public fun securityStateChanged(): Flow<SecurityEvent.SecurityStateChanged> =
      securityStateChangedEvents()

  /**
   * Disables tracking security state changes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("Security.disable", Unit)

  /**
   * Enables tracking security state changes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("Security.enable", Unit)

  /**
   * Enable/disable whether all certificate errors should be ignored.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-setIgnoreCertificateErrors)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setIgnoreCertificateErrors(input: SetIgnoreCertificateErrorsRequest):
      SetIgnoreCertificateErrorsResponse = session.request("Security.setIgnoreCertificateErrors",
      input)

  /**
   * Enable/disable whether all certificate errors should be ignored.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-setIgnoreCertificateErrors)
   *
   * @param ignore If true, all certificate errors will be ignored.
   */
  @ExperimentalChromeApi
  public suspend fun setIgnoreCertificateErrors(ignore: Boolean):
      SetIgnoreCertificateErrorsResponse {
    val input = SetIgnoreCertificateErrorsRequest(ignore)
    return setIgnoreCertificateErrors(input)
  }

  /**
   * Handles a certificate error that fired a certificateError event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-handleCertificateError)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun handleCertificateError(input: HandleCertificateErrorRequest):
      HandleCertificateErrorResponse = session.request("Security.handleCertificateError", input)

  /**
   * Handles a certificate error that fired a certificateError event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-handleCertificateError)
   *
   * @param eventId The ID of the event.
   * @param action The action to take on the certificate error.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun handleCertificateError(eventId: Int, action: CertificateErrorAction):
      HandleCertificateErrorResponse {
    val input = HandleCertificateErrorRequest(eventId, action)
    return handleCertificateError(input)
  }

  /**
   * Enable/disable overriding certificate errors. If enabled, all certificate error events need to
   * be handled by the DevTools client and should be answered with `handleCertificateError`
   * commands.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-setOverrideCertificateErrors)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun setOverrideCertificateErrors(input: SetOverrideCertificateErrorsRequest):
      SetOverrideCertificateErrorsResponse =
      session.request("Security.setOverrideCertificateErrors", input)

  /**
   * Enable/disable overriding certificate errors. If enabled, all certificate error events need to
   * be handled by the DevTools client and should be answered with `handleCertificateError`
   * commands.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#method-setOverrideCertificateErrors)
   *
   * @param override If true, certificate errors will be overridden.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun setOverrideCertificateErrors(`override`: Boolean):
      SetOverrideCertificateErrorsResponse {
    val input = SetOverrideCertificateErrorsRequest(override)
    return setOverrideCertificateErrors(input)
  }
}
