@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.security

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * An internal certificate ID value.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-CertificateId)
 */
public typealias CertificateId = Int

/**
 * A description of mixed content (HTTP resources on HTTPS pages), as defined by
 * https://www.w3.org/TR/mixed-content/#categories
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-MixedContentType)
 */
@Serializable
public enum class MixedContentType {
  blockable,
  `optionally-blockable`,
  none,
}

/**
 * The security level of a page or resource.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-SecurityState)
 */
@Serializable
public enum class SecurityState {
  unknown,
  neutral,
  insecure,
  secure,
  info,
  `insecure-broken`,
}

/**
 * Details about the security state of the page certificate.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-CertificateSecurityState)
 */
@ExperimentalChromeApi
@Serializable
public data class CertificateSecurityState(
  /**
   * Protocol name (e.g. "TLS 1.2" or "QUIC").
   */
  public val protocol: String,
  /**
   * Key Exchange used by the connection, or the empty string if not applicable.
   */
  public val keyExchange: String,
  /**
   * (EC)DH group used by the connection, if applicable.
   */
  public val keyExchangeGroup: String? = null,
  /**
   * Cipher name.
   */
  public val cipher: String,
  /**
   * TLS MAC. Note that AEAD ciphers do not have separate MACs.
   */
  public val mac: String? = null,
  /**
   * Page certificate.
   */
  public val certificate: List<String>,
  /**
   * Certificate subject name.
   */
  public val subjectName: String,
  /**
   * Name of the issuing CA.
   */
  public val issuer: String,
  /**
   * Certificate valid from date.
   */
  public val validFrom: TimeSinceEpoch,
  /**
   * Certificate valid to (expiration) date
   */
  public val validTo: TimeSinceEpoch,
  /**
   * The highest priority network error code, if the certificate has an error.
   */
  public val certificateNetworkError: String? = null,
  /**
   * True if the certificate uses a weak signature aglorithm.
   */
  public val certificateHasWeakSignature: Boolean,
  /**
   * True if the certificate has a SHA1 signature in the chain.
   */
  public val certificateHasSha1Signature: Boolean,
  /**
   * True if modern SSL
   */
  public val modernSSL: Boolean,
  /**
   * True if the connection is using an obsolete SSL protocol.
   */
  public val obsoleteSslProtocol: Boolean,
  /**
   * True if the connection is using an obsolete SSL key exchange.
   */
  public val obsoleteSslKeyExchange: Boolean,
  /**
   * True if the connection is using an obsolete SSL cipher.
   */
  public val obsoleteSslCipher: Boolean,
  /**
   * True if the connection is using an obsolete SSL signature.
   */
  public val obsoleteSslSignature: Boolean,
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-SafetyTipStatus)
 */
@ExperimentalChromeApi
@Serializable
public enum class SafetyTipStatus {
  badReputation,
  lookalike,
}

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-SafetyTipInfo)
 */
@ExperimentalChromeApi
@Serializable
public data class SafetyTipInfo(
  /**
   * Describes whether the page triggers any safety tips or reputation warnings. Default is unknown.
   */
  public val safetyTipStatus: SafetyTipStatus,
  /**
   * The URL the safety tip suggested ("Did you mean?"). Only filled in for lookalike matches.
   */
  public val safeUrl: String? = null,
)

/**
 * Security state information about the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-VisibleSecurityState)
 */
@ExperimentalChromeApi
@Serializable
public data class VisibleSecurityState(
  /**
   * The security level of the page.
   */
  public val securityState: SecurityState,
  /**
   * Security state details about the page certificate.
   */
  public val certificateSecurityState: CertificateSecurityState? = null,
  /**
   * The type of Safety Tip triggered on the page. Note that this field will be set even if the
   * Safety Tip UI was not actually shown.
   */
  public val safetyTipInfo: SafetyTipInfo? = null,
  /**
   * Array of security state issues ids.
   */
  public val securityStateIssueIds: List<String>,
)

/**
 * An explanation of an factor contributing to the security state.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-SecurityStateExplanation)
 */
@Serializable
public data class SecurityStateExplanation(
  /**
   * Security state representing the severity of the factor being explained.
   */
  public val securityState: SecurityState,
  /**
   * Title describing the type of factor.
   */
  public val title: String,
  /**
   * Short phrase describing the type of factor.
   */
  public val summary: String,
  /**
   * Full text explanation of the factor.
   */
  public val description: String,
  /**
   * The type of mixed content described by the explanation.
   */
  public val mixedContentType: MixedContentType,
  /**
   * Page certificate.
   */
  public val certificate: List<String>,
  /**
   * Recommendations to fix any issues.
   */
  public val recommendations: List<String>? = null,
)

/**
 * Information about insecure content on the page.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-InsecureContentStatus)
 */
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
@Serializable
public data class InsecureContentStatus(
  /**
   * Always false.
   */
  public val ranMixedContent: Boolean,
  /**
   * Always false.
   */
  public val displayedMixedContent: Boolean,
  /**
   * Always false.
   */
  public val containedMixedForm: Boolean,
  /**
   * Always false.
   */
  public val ranContentWithCertErrors: Boolean,
  /**
   * Always false.
   */
  public val displayedContentWithCertErrors: Boolean,
  /**
   * Always set to unknown.
   */
  public val ranInsecureContentStyle: SecurityState,
  /**
   * Always set to unknown.
   */
  public val displayedInsecureContentStyle: SecurityState,
)

/**
 * The action to take when a certificate error occurs. continue will continue processing the
 * request and cancel will cancel the request.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Security/#type-CertificateErrorAction)
 */
@Serializable
public enum class CertificateErrorAction {
  `continue`,
  cancel,
}
