@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.storage

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.browser.BrowserContextID
import org.hildan.chrome.devtools.domains.network.Cookie
import org.hildan.chrome.devtools.domains.network.CookieParam
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.storage.events.StorageEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [StorageDomain.getStorageKeyForFrame] command.
 */
@Serializable
public data class GetStorageKeyForFrameRequest(
  public val frameId: FrameId,
)

/**
 * Response type for the [StorageDomain.getStorageKeyForFrame] command.
 */
@Serializable
public data class GetStorageKeyForFrameResponse(
  public val storageKey: SerializedStorageKey,
)

/**
 * Request object containing input parameters for the [StorageDomain.clearDataForOrigin] command.
 */
@Serializable
public data class ClearDataForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
  /**
   * Comma separated list of StorageType to clear.
   */
  public val storageTypes: String,
)

/**
 * A dummy response object for the [StorageDomain.clearDataForOrigin] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ClearDataForOriginResponse

/**
 * Request object containing input parameters for the [StorageDomain.clearDataForStorageKey]
 * command.
 */
@Serializable
public data class ClearDataForStorageKeyRequest(
  /**
   * Storage key.
   */
  public val storageKey: String,
  /**
   * Comma separated list of StorageType to clear.
   */
  public val storageTypes: String,
)

/**
 * A dummy response object for the [StorageDomain.clearDataForStorageKey] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object ClearDataForStorageKeyResponse

/**
 * Request object containing input parameters for the [StorageDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesRequest(
  /**
   * Browser context to use when called on the browser endpoint.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder() {
    /**
     * Browser context to use when called on the browser endpoint.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): GetCookiesRequest = GetCookiesRequest(browserContextId)
  }
}

/**
 * Response type for the [StorageDomain.getCookies] command.
 */
@Serializable
public data class GetCookiesResponse(
  /**
   * Array of cookie objects.
   */
  public val cookies: List<Cookie>,
)

/**
 * Request object containing input parameters for the [StorageDomain.setCookies] command.
 */
@Serializable
public data class SetCookiesRequest(
  /**
   * Cookies to be set.
   */
  public val cookies: List<CookieParam>,
  /**
   * Browser context to use when called on the browser endpoint.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder(
    /**
     * Cookies to be set.
     */
    public val cookies: List<CookieParam>,
  ) {
    /**
     * Browser context to use when called on the browser endpoint.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): SetCookiesRequest = SetCookiesRequest(cookies, browserContextId)
  }
}

/**
 * A dummy response object for the [StorageDomain.setCookies] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object SetCookiesResponse

/**
 * Request object containing input parameters for the [StorageDomain.clearCookies] command.
 */
@Serializable
public data class ClearCookiesRequest(
  /**
   * Browser context to use when called on the browser endpoint.
   */
  public val browserContextId: BrowserContextID? = null,
) {
  public class Builder() {
    /**
     * Browser context to use when called on the browser endpoint.
     */
    public var browserContextId: BrowserContextID? = null

    public fun build(): ClearCookiesRequest = ClearCookiesRequest(browserContextId)
  }
}

/**
 * A dummy response object for the [StorageDomain.clearCookies] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
public object ClearCookiesResponse

/**
 * Request object containing input parameters for the [StorageDomain.getUsageAndQuota] command.
 */
@Serializable
public data class GetUsageAndQuotaRequest(
  /**
   * Security origin.
   */
  public val origin: String,
)

/**
 * Response type for the [StorageDomain.getUsageAndQuota] command.
 */
@Serializable
public data class GetUsageAndQuotaResponse(
  /**
   * Storage usage (bytes).
   */
  public val usage: Double,
  /**
   * Storage quota (bytes).
   */
  public val quota: Double,
  /**
   * Whether or not the origin has an active storage quota override
   */
  public val overrideActive: Boolean,
  /**
   * Storage usage per type (bytes).
   */
  public val usageBreakdown: List<UsageForType>,
)

/**
 * Request object containing input parameters for the [StorageDomain.overrideQuotaForOrigin]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class OverrideQuotaForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
  /**
   * The quota size (in bytes) to override the original quota with.
   * If this is called multiple times, the overridden quota will be equal to
   * the quotaSize provided in the final call. If this is called without
   * specifying a quotaSize, the quota will be reset to the default value for
   * the specified origin. If this is called multiple times with different
   * origins, the override will be maintained for each origin until it is
   * disabled (called without a quotaSize).
   */
  public val quotaSize: Double? = null,
) {
  public class Builder(
    /**
     * Security origin.
     */
    public val origin: String,
  ) {
    /**
     * The quota size (in bytes) to override the original quota with.
     * If this is called multiple times, the overridden quota will be equal to
     * the quotaSize provided in the final call. If this is called without
     * specifying a quotaSize, the quota will be reset to the default value for
     * the specified origin. If this is called multiple times with different
     * origins, the override will be maintained for each origin until it is
     * disabled (called without a quotaSize).
     */
    public var quotaSize: Double? = null

    public fun build(): OverrideQuotaForOriginRequest = OverrideQuotaForOriginRequest(origin,
        quotaSize)
  }
}

/**
 * A dummy response object for the [StorageDomain.overrideQuotaForOrigin] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object OverrideQuotaForOriginResponse

/**
 * Request object containing input parameters for the [StorageDomain.trackCacheStorageForOrigin]
 * command.
 */
@Serializable
public data class TrackCacheStorageForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
)

/**
 * A dummy response object for the [StorageDomain.trackCacheStorageForOrigin] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object TrackCacheStorageForOriginResponse

/**
 * Request object containing input parameters for the [StorageDomain.trackIndexedDBForOrigin]
 * command.
 */
@Serializable
public data class TrackIndexedDBForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
)

/**
 * A dummy response object for the [StorageDomain.trackIndexedDBForOrigin] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object TrackIndexedDBForOriginResponse

/**
 * Request object containing input parameters for the [StorageDomain.trackIndexedDBForStorageKey]
 * command.
 */
@Serializable
public data class TrackIndexedDBForStorageKeyRequest(
  /**
   * Storage key.
   */
  public val storageKey: String,
)

/**
 * A dummy response object for the [StorageDomain.trackIndexedDBForStorageKey] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object TrackIndexedDBForStorageKeyResponse

/**
 * Request object containing input parameters for the [StorageDomain.untrackCacheStorageForOrigin]
 * command.
 */
@Serializable
public data class UntrackCacheStorageForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
)

/**
 * A dummy response object for the [StorageDomain.untrackCacheStorageForOrigin] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object UntrackCacheStorageForOriginResponse

/**
 * Request object containing input parameters for the [StorageDomain.untrackIndexedDBForOrigin]
 * command.
 */
@Serializable
public data class UntrackIndexedDBForOriginRequest(
  /**
   * Security origin.
   */
  public val origin: String,
)

/**
 * A dummy response object for the [StorageDomain.untrackIndexedDBForOrigin] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object UntrackIndexedDBForOriginResponse

/**
 * Request object containing input parameters for the [StorageDomain.untrackIndexedDBForStorageKey]
 * command.
 */
@Serializable
public data class UntrackIndexedDBForStorageKeyRequest(
  /**
   * Storage key.
   */
  public val storageKey: String,
)

/**
 * A dummy response object for the [StorageDomain.untrackIndexedDBForStorageKey] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object UntrackIndexedDBForStorageKeyResponse

/**
 * Response type for the [StorageDomain.getTrustTokens] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetTrustTokensResponse(
  public val tokens: List<TrustTokens>,
)

/**
 * Request object containing input parameters for the [StorageDomain.clearTrustTokens] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ClearTrustTokensRequest(
  public val issuerOrigin: String,
)

/**
 * Response type for the [StorageDomain.clearTrustTokens] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ClearTrustTokensResponse(
  /**
   * True if any tokens were deleted, false otherwise.
   */
  public val didDeleteTokens: Boolean,
)

/**
 * Request object containing input parameters for the [StorageDomain.getInterestGroupDetails]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetInterestGroupDetailsRequest(
  public val ownerOrigin: String,
  public val name: String,
)

/**
 * Response type for the [StorageDomain.getInterestGroupDetails] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetInterestGroupDetailsResponse(
  public val details: InterestGroupDetails,
)

/**
 * Request object containing input parameters for the [StorageDomain.setInterestGroupTracking]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetInterestGroupTrackingRequest(
  public val enable: Boolean,
)

/**
 * A dummy response object for the [StorageDomain.setInterestGroupTracking] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetInterestGroupTrackingResponse

/**
 * Request object containing input parameters for the [StorageDomain.getSharedStorageMetadata]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSharedStorageMetadataRequest(
  public val ownerOrigin: String,
)

/**
 * Response type for the [StorageDomain.getSharedStorageMetadata] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSharedStorageMetadataResponse(
  public val metadata: SharedStorageMetadata,
)

/**
 * Request object containing input parameters for the [StorageDomain.getSharedStorageEntries]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSharedStorageEntriesRequest(
  public val ownerOrigin: String,
)

/**
 * Response type for the [StorageDomain.getSharedStorageEntries] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetSharedStorageEntriesResponse(
  public val entries: List<SharedStorageEntry>,
)

/**
 * Request object containing input parameters for the [StorageDomain.deleteSharedStorageEntry]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class DeleteSharedStorageEntryRequest(
  public val ownerOrigin: String,
  public val key: String,
)

/**
 * A dummy response object for the [StorageDomain.deleteSharedStorageEntry] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object DeleteSharedStorageEntryResponse

/**
 * Request object containing input parameters for the [StorageDomain.clearSharedStorageEntries]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class ClearSharedStorageEntriesRequest(
  public val ownerOrigin: String,
)

/**
 * A dummy response object for the [StorageDomain.clearSharedStorageEntries] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ClearSharedStorageEntriesResponse

/**
 * Request object containing input parameters for the [StorageDomain.setSharedStorageTracking]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetSharedStorageTrackingRequest(
  public val enable: Boolean,
)

/**
 * A dummy response object for the [StorageDomain.setSharedStorageTracking] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetSharedStorageTrackingResponse

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage)
 */
@ExperimentalChromeApi
public class StorageDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out StorageEvent>> =
      mapOf(
      "Storage.cacheStorageContentUpdated" to serializer<StorageEvent.CacheStorageContentUpdated>(),
      "Storage.cacheStorageListUpdated" to serializer<StorageEvent.CacheStorageListUpdated>(),
      "Storage.indexedDBContentUpdated" to serializer<StorageEvent.IndexedDBContentUpdated>(),
      "Storage.indexedDBListUpdated" to serializer<StorageEvent.IndexedDBListUpdated>(),
      "Storage.interestGroupAccessed" to serializer<StorageEvent.InterestGroupAccessed>(),
      "Storage.sharedStorageAccessed" to serializer<StorageEvent.SharedStorageAccessed>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<StorageEvent> = session.typedEvents(deserializersByEventName)

  /**
   * A cache's contents have been modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-cacheStorageContentUpdated)
   */
  public fun cacheStorageContentUpdatedEvents(): Flow<StorageEvent.CacheStorageContentUpdated> =
      session.typedEvents("Storage.cacheStorageContentUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("cacheStorageContentUpdatedEvents()"),
  )
  public fun cacheStorageContentUpdated(): Flow<StorageEvent.CacheStorageContentUpdated> =
      cacheStorageContentUpdatedEvents()

  /**
   * A cache has been added/deleted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-cacheStorageListUpdated)
   */
  public fun cacheStorageListUpdatedEvents(): Flow<StorageEvent.CacheStorageListUpdated> =
      session.typedEvents("Storage.cacheStorageListUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("cacheStorageListUpdatedEvents()"),
  )
  public fun cacheStorageListUpdated(): Flow<StorageEvent.CacheStorageListUpdated> =
      cacheStorageListUpdatedEvents()

  /**
   * The origin's IndexedDB object store has been modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-indexedDBContentUpdated)
   */
  public fun indexedDBContentUpdatedEvents(): Flow<StorageEvent.IndexedDBContentUpdated> =
      session.typedEvents("Storage.indexedDBContentUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("indexedDBContentUpdatedEvents()"),
  )
  public fun indexedDBContentUpdated(): Flow<StorageEvent.IndexedDBContentUpdated> =
      indexedDBContentUpdatedEvents()

  /**
   * The origin's IndexedDB database list has been modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-indexedDBListUpdated)
   */
  public fun indexedDBListUpdatedEvents(): Flow<StorageEvent.IndexedDBListUpdated> =
      session.typedEvents("Storage.indexedDBListUpdated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("indexedDBListUpdatedEvents()"),
  )
  public fun indexedDBListUpdated(): Flow<StorageEvent.IndexedDBListUpdated> =
      indexedDBListUpdatedEvents()

  /**
   * One of the interest groups was accessed by the associated page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-interestGroupAccessed)
   */
  public fun interestGroupAccessedEvents(): Flow<StorageEvent.InterestGroupAccessed> =
      session.typedEvents("Storage.interestGroupAccessed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("interestGroupAccessedEvents()"),
  )
  public fun interestGroupAccessed(): Flow<StorageEvent.InterestGroupAccessed> =
      interestGroupAccessedEvents()

  /**
   * Shared storage was accessed by the associated page.
   * The following parameters are included in all events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-sharedStorageAccessed)
   */
  public fun sharedStorageAccessedEvents(): Flow<StorageEvent.SharedStorageAccessed> =
      session.typedEvents("Storage.sharedStorageAccessed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("sharedStorageAccessedEvents()"),
  )
  public fun sharedStorageAccessed(): Flow<StorageEvent.SharedStorageAccessed> =
      sharedStorageAccessedEvents()

  /**
   * Returns a storage key given a frame id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getStorageKeyForFrame)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getStorageKeyForFrame(input: GetStorageKeyForFrameRequest):
      GetStorageKeyForFrameResponse = session.request("Storage.getStorageKeyForFrame", input)

  /**
   * Returns a storage key given a frame id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getStorageKeyForFrame)
   */
  public suspend fun getStorageKeyForFrame(frameId: FrameId): GetStorageKeyForFrameResponse {
    val input = GetStorageKeyForFrameRequest(frameId)
    return getStorageKeyForFrame(input)
  }

  /**
   * Clears storage for origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearDataForOrigin)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clearDataForOrigin(input: ClearDataForOriginRequest):
      ClearDataForOriginResponse = session.request("Storage.clearDataForOrigin", input)

  /**
   * Clears storage for origin.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearDataForOrigin)
   *
   * @param origin Security origin.
   * @param storageTypes Comma separated list of StorageType to clear.
   */
  public suspend fun clearDataForOrigin(origin: String, storageTypes: String):
      ClearDataForOriginResponse {
    val input = ClearDataForOriginRequest(origin, storageTypes)
    return clearDataForOrigin(input)
  }

  /**
   * Clears storage for storage key.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearDataForStorageKey)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clearDataForStorageKey(input: ClearDataForStorageKeyRequest):
      ClearDataForStorageKeyResponse = session.request("Storage.clearDataForStorageKey", input)

  /**
   * Clears storage for storage key.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearDataForStorageKey)
   *
   * @param storageKey Storage key.
   * @param storageTypes Comma separated list of StorageType to clear.
   */
  public suspend fun clearDataForStorageKey(storageKey: String, storageTypes: String):
      ClearDataForStorageKeyResponse {
    val input = ClearDataForStorageKeyRequest(storageKey, storageTypes)
    return clearDataForStorageKey(input)
  }

  /**
   * Returns all browser cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getCookies)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getCookies(input: GetCookiesRequest): GetCookiesResponse =
      session.request("Storage.getCookies", input)

  /**
   * Returns all browser cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getCookies)
   */
  @JvmOverloads
  public suspend inline fun getCookies(optionalArgs: GetCookiesRequest.Builder.() -> Unit = {}):
      GetCookiesResponse {
    val builder = GetCookiesRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getCookies(input)
  }

  /**
   * Sets given cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-setCookies)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setCookies(input: SetCookiesRequest): SetCookiesResponse =
      session.request("Storage.setCookies", input)

  /**
   * Sets given cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-setCookies)
   *
   * @param cookies Cookies to be set.
   */
  @JvmOverloads
  public suspend inline fun setCookies(cookies: List<CookieParam>,
      optionalArgs: SetCookiesRequest.Builder.() -> Unit = {}): SetCookiesResponse {
    val builder = SetCookiesRequest.Builder(cookies)
    val input = builder.apply(optionalArgs).build()
    return setCookies(input)
  }

  /**
   * Clears cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearCookies)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clearCookies(input: ClearCookiesRequest): ClearCookiesResponse =
      session.request("Storage.clearCookies", input)

  /**
   * Clears cookies.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearCookies)
   */
  @JvmOverloads
  public suspend inline fun clearCookies(optionalArgs: ClearCookiesRequest.Builder.() -> Unit = {}):
      ClearCookiesResponse {
    val builder = ClearCookiesRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return clearCookies(input)
  }

  /**
   * Returns usage and quota in bytes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getUsageAndQuota)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getUsageAndQuota(input: GetUsageAndQuotaRequest): GetUsageAndQuotaResponse =
      session.request("Storage.getUsageAndQuota", input)

  /**
   * Returns usage and quota in bytes.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getUsageAndQuota)
   *
   * @param origin Security origin.
   */
  public suspend fun getUsageAndQuota(origin: String): GetUsageAndQuotaResponse {
    val input = GetUsageAndQuotaRequest(origin)
    return getUsageAndQuota(input)
  }

  /**
   * Override quota for the specified origin
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-overrideQuotaForOrigin)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun overrideQuotaForOrigin(input: OverrideQuotaForOriginRequest):
      OverrideQuotaForOriginResponse = session.request("Storage.overrideQuotaForOrigin", input)

  /**
   * Override quota for the specified origin
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-overrideQuotaForOrigin)
   *
   * @param origin Security origin.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun overrideQuotaForOrigin(origin: String,
      optionalArgs: OverrideQuotaForOriginRequest.Builder.() -> Unit = {}):
      OverrideQuotaForOriginResponse {
    val builder = OverrideQuotaForOriginRequest.Builder(origin)
    val input = builder.apply(optionalArgs).build()
    return overrideQuotaForOrigin(input)
  }

  /**
   * Registers origin to be notified when an update occurs to its cache storage list.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-trackCacheStorageForOrigin)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun trackCacheStorageForOrigin(input: TrackCacheStorageForOriginRequest):
      TrackCacheStorageForOriginResponse = session.request("Storage.trackCacheStorageForOrigin",
      input)

  /**
   * Registers origin to be notified when an update occurs to its cache storage list.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-trackCacheStorageForOrigin)
   *
   * @param origin Security origin.
   */
  public suspend fun trackCacheStorageForOrigin(origin: String):
      TrackCacheStorageForOriginResponse {
    val input = TrackCacheStorageForOriginRequest(origin)
    return trackCacheStorageForOrigin(input)
  }

  /**
   * Registers origin to be notified when an update occurs to its IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-trackIndexedDBForOrigin)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun trackIndexedDBForOrigin(input: TrackIndexedDBForOriginRequest):
      TrackIndexedDBForOriginResponse = session.request("Storage.trackIndexedDBForOrigin", input)

  /**
   * Registers origin to be notified when an update occurs to its IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-trackIndexedDBForOrigin)
   *
   * @param origin Security origin.
   */
  public suspend fun trackIndexedDBForOrigin(origin: String): TrackIndexedDBForOriginResponse {
    val input = TrackIndexedDBForOriginRequest(origin)
    return trackIndexedDBForOrigin(input)
  }

  /**
   * Registers storage key to be notified when an update occurs to its IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-trackIndexedDBForStorageKey)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun trackIndexedDBForStorageKey(input: TrackIndexedDBForStorageKeyRequest):
      TrackIndexedDBForStorageKeyResponse = session.request("Storage.trackIndexedDBForStorageKey",
      input)

  /**
   * Registers storage key to be notified when an update occurs to its IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-trackIndexedDBForStorageKey)
   *
   * @param storageKey Storage key.
   */
  public suspend fun trackIndexedDBForStorageKey(storageKey: String):
      TrackIndexedDBForStorageKeyResponse {
    val input = TrackIndexedDBForStorageKeyRequest(storageKey)
    return trackIndexedDBForStorageKey(input)
  }

  /**
   * Unregisters origin from receiving notifications for cache storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-untrackCacheStorageForOrigin)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun untrackCacheStorageForOrigin(input: UntrackCacheStorageForOriginRequest):
      UntrackCacheStorageForOriginResponse = session.request("Storage.untrackCacheStorageForOrigin",
      input)

  /**
   * Unregisters origin from receiving notifications for cache storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-untrackCacheStorageForOrigin)
   *
   * @param origin Security origin.
   */
  public suspend fun untrackCacheStorageForOrigin(origin: String):
      UntrackCacheStorageForOriginResponse {
    val input = UntrackCacheStorageForOriginRequest(origin)
    return untrackCacheStorageForOrigin(input)
  }

  /**
   * Unregisters origin from receiving notifications for IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-untrackIndexedDBForOrigin)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun untrackIndexedDBForOrigin(input: UntrackIndexedDBForOriginRequest):
      UntrackIndexedDBForOriginResponse = session.request("Storage.untrackIndexedDBForOrigin",
      input)

  /**
   * Unregisters origin from receiving notifications for IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-untrackIndexedDBForOrigin)
   *
   * @param origin Security origin.
   */
  public suspend fun untrackIndexedDBForOrigin(origin: String): UntrackIndexedDBForOriginResponse {
    val input = UntrackIndexedDBForOriginRequest(origin)
    return untrackIndexedDBForOrigin(input)
  }

  /**
   * Unregisters storage key from receiving notifications for IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-untrackIndexedDBForStorageKey)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun untrackIndexedDBForStorageKey(input: UntrackIndexedDBForStorageKeyRequest):
      UntrackIndexedDBForStorageKeyResponse =
      session.request("Storage.untrackIndexedDBForStorageKey", input)

  /**
   * Unregisters storage key from receiving notifications for IndexedDB.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-untrackIndexedDBForStorageKey)
   *
   * @param storageKey Storage key.
   */
  public suspend fun untrackIndexedDBForStorageKey(storageKey: String):
      UntrackIndexedDBForStorageKeyResponse {
    val input = UntrackIndexedDBForStorageKeyRequest(storageKey)
    return untrackIndexedDBForStorageKey(input)
  }

  /**
   * Returns the number of stored Trust Tokens per issuer for the
   * current browsing context.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getTrustTokens)
   */
  @ExperimentalChromeApi
  public suspend fun getTrustTokens(): GetTrustTokensResponse =
      session.request("Storage.getTrustTokens", Unit)

  /**
   * Removes all Trust Tokens issued by the provided issuerOrigin.
   * Leaves other stored data, including the issuer's Redemption Records, intact.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearTrustTokens)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun clearTrustTokens(input: ClearTrustTokensRequest): ClearTrustTokensResponse =
      session.request("Storage.clearTrustTokens", input)

  /**
   * Removes all Trust Tokens issued by the provided issuerOrigin.
   * Leaves other stored data, including the issuer's Redemption Records, intact.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearTrustTokens)
   */
  @ExperimentalChromeApi
  public suspend fun clearTrustTokens(issuerOrigin: String): ClearTrustTokensResponse {
    val input = ClearTrustTokensRequest(issuerOrigin)
    return clearTrustTokens(input)
  }

  /**
   * Gets details for a named interest group.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getInterestGroupDetails)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getInterestGroupDetails(input: GetInterestGroupDetailsRequest):
      GetInterestGroupDetailsResponse = session.request("Storage.getInterestGroupDetails", input)

  /**
   * Gets details for a named interest group.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getInterestGroupDetails)
   */
  @ExperimentalChromeApi
  public suspend fun getInterestGroupDetails(ownerOrigin: String, name: String):
      GetInterestGroupDetailsResponse {
    val input = GetInterestGroupDetailsRequest(ownerOrigin, name)
    return getInterestGroupDetails(input)
  }

  /**
   * Enables/Disables issuing of interestGroupAccessed events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-setInterestGroupTracking)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setInterestGroupTracking(input: SetInterestGroupTrackingRequest):
      SetInterestGroupTrackingResponse = session.request("Storage.setInterestGroupTracking", input)

  /**
   * Enables/Disables issuing of interestGroupAccessed events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-setInterestGroupTracking)
   */
  @ExperimentalChromeApi
  public suspend fun setInterestGroupTracking(enable: Boolean): SetInterestGroupTrackingResponse {
    val input = SetInterestGroupTrackingRequest(enable)
    return setInterestGroupTracking(input)
  }

  /**
   * Gets metadata for an origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getSharedStorageMetadata)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getSharedStorageMetadata(input: GetSharedStorageMetadataRequest):
      GetSharedStorageMetadataResponse = session.request("Storage.getSharedStorageMetadata", input)

  /**
   * Gets metadata for an origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getSharedStorageMetadata)
   */
  @ExperimentalChromeApi
  public suspend fun getSharedStorageMetadata(ownerOrigin: String):
      GetSharedStorageMetadataResponse {
    val input = GetSharedStorageMetadataRequest(ownerOrigin)
    return getSharedStorageMetadata(input)
  }

  /**
   * Gets the entries in an given origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getSharedStorageEntries)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getSharedStorageEntries(input: GetSharedStorageEntriesRequest):
      GetSharedStorageEntriesResponse = session.request("Storage.getSharedStorageEntries", input)

  /**
   * Gets the entries in an given origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-getSharedStorageEntries)
   */
  @ExperimentalChromeApi
  public suspend fun getSharedStorageEntries(ownerOrigin: String): GetSharedStorageEntriesResponse {
    val input = GetSharedStorageEntriesRequest(ownerOrigin)
    return getSharedStorageEntries(input)
  }

  /**
   * Deletes entry for `key` (if it exists) for a given origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-deleteSharedStorageEntry)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun deleteSharedStorageEntry(input: DeleteSharedStorageEntryRequest):
      DeleteSharedStorageEntryResponse = session.request("Storage.deleteSharedStorageEntry", input)

  /**
   * Deletes entry for `key` (if it exists) for a given origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-deleteSharedStorageEntry)
   */
  @ExperimentalChromeApi
  public suspend fun deleteSharedStorageEntry(ownerOrigin: String, key: String):
      DeleteSharedStorageEntryResponse {
    val input = DeleteSharedStorageEntryRequest(ownerOrigin, key)
    return deleteSharedStorageEntry(input)
  }

  /**
   * Clears all entries for a given origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearSharedStorageEntries)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun clearSharedStorageEntries(input: ClearSharedStorageEntriesRequest):
      ClearSharedStorageEntriesResponse = session.request("Storage.clearSharedStorageEntries",
      input)

  /**
   * Clears all entries for a given origin's shared storage.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-clearSharedStorageEntries)
   */
  @ExperimentalChromeApi
  public suspend fun clearSharedStorageEntries(ownerOrigin: String):
      ClearSharedStorageEntriesResponse {
    val input = ClearSharedStorageEntriesRequest(ownerOrigin)
    return clearSharedStorageEntries(input)
  }

  /**
   * Enables/disables issuing of sharedStorageAccessed events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-setSharedStorageTracking)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setSharedStorageTracking(input: SetSharedStorageTrackingRequest):
      SetSharedStorageTrackingResponse = session.request("Storage.setSharedStorageTracking", input)

  /**
   * Enables/disables issuing of sharedStorageAccessed events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#method-setSharedStorageTracking)
   */
  @ExperimentalChromeApi
  public suspend fun setSharedStorageTracking(enable: Boolean): SetSharedStorageTrackingResponse {
    val input = SetSharedStorageTrackingRequest(enable)
    return setSharedStorageTracking(input)
  }
}
