@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.storage

import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SerializedStorageKey)
 */
public typealias SerializedStorageKey = String

/**
 * Enum of possible storage types.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-StorageType)
 */
@Serializable
public enum class StorageType {
  appcache,
  cookies,
  file_systems,
  indexeddb,
  local_storage,
  shader_cache,
  websql,
  service_workers,
  cache_storage,
  interest_groups,
  shared_storage,
  all,
  other,
}

/**
 * Usage for a storage type.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-UsageForType)
 */
@Serializable
public data class UsageForType(
  /**
   * Name of storage type.
   */
  public val storageType: StorageType,
  /**
   * Storage usage (bytes).
   */
  public val usage: Double,
)

/**
 * Pair of issuer origin and number of available (signed, but not used) Trust
 * Tokens from that issuer.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-TrustTokens)
 */
@ExperimentalChromeApi
@Serializable
public data class TrustTokens(
  public val issuerOrigin: String,
  public val count: Double,
)

/**
 * Enum of interest group access types.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-InterestGroupAccessType)
 */
@Serializable
public enum class InterestGroupAccessType {
  join,
  leave,
  update,
  bid,
  win,
}

/**
 * Ad advertising element inside an interest group.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-InterestGroupAd)
 */
@Serializable
public data class InterestGroupAd(
  public val renderUrl: String,
  public val metadata: String? = null,
)

/**
 * The full details of an interest group.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-InterestGroupDetails)
 */
@Serializable
public data class InterestGroupDetails(
  public val ownerOrigin: String,
  public val name: String,
  public val expirationTime: TimeSinceEpoch,
  public val joiningOrigin: String,
  public val biddingUrl: String? = null,
  public val biddingWasmHelperUrl: String? = null,
  public val updateUrl: String? = null,
  public val trustedBiddingSignalsUrl: String? = null,
  public val trustedBiddingSignalsKeys: List<String>,
  public val userBiddingSignals: String? = null,
  public val ads: List<InterestGroupAd>,
  public val adComponents: List<InterestGroupAd>,
)

/**
 * Enum of shared storage access types.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SharedStorageAccessType)
 */
@Serializable
public enum class SharedStorageAccessType {
  documentAddModule,
  documentSelectURL,
  documentRun,
  documentSet,
  documentAppend,
  documentDelete,
  documentClear,
  workletSet,
  workletAppend,
  workletDelete,
  workletClear,
  workletGet,
  workletKeys,
  workletEntries,
  workletLength,
  workletRemainingBudget,
}

/**
 * Struct for a single key-value pair in an origin's shared storage.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SharedStorageEntry)
 */
@Serializable
public data class SharedStorageEntry(
  public val key: String,
  public val `value`: String,
)

/**
 * Details for an origin's shared storage.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SharedStorageMetadata)
 */
@Serializable
public data class SharedStorageMetadata(
  public val creationTime: TimeSinceEpoch,
  public val length: Int,
  public val remainingBudget: Double,
)

/**
 * Pair of reporting metadata details for a candidate URL for `selectURL()`.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SharedStorageReportingMetadata)
 */
@Serializable
public data class SharedStorageReportingMetadata(
  public val eventType: String,
  public val reportingUrl: String,
)

/**
 * Bundles a candidate URL with its reporting metadata.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SharedStorageUrlWithMetadata)
 */
@Serializable
public data class SharedStorageUrlWithMetadata(
  /**
   * Spec of candidate URL.
   */
  public val url: String,
  /**
   * Any associated reporting metadata.
   */
  public val reportingMetadata: List<SharedStorageReportingMetadata>,
)

/**
 * Bundles the parameters for shared storage access events whose
 * presence/absence can vary according to SharedStorageAccessType.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#type-SharedStorageAccessParams)
 */
@Serializable
public data class SharedStorageAccessParams(
  /**
   * Spec of the module script URL.
   * Present only for SharedStorageAccessType.documentAddModule.
   */
  public val scriptSourceUrl: String? = null,
  /**
   * Name of the registered operation to be run.
   * Present only for SharedStorageAccessType.documentRun and
   * SharedStorageAccessType.documentSelectURL.
   */
  public val operationName: String? = null,
  /**
   * The operation's serialized data in bytes (converted to a string).
   * Present only for SharedStorageAccessType.documentRun and
   * SharedStorageAccessType.documentSelectURL.
   */
  public val serializedData: String? = null,
  /**
   * Array of candidate URLs' specs, along with any associated metadata.
   * Present only for SharedStorageAccessType.documentSelectURL.
   */
  public val urlsWithMetadata: List<SharedStorageUrlWithMetadata>? = null,
  /**
   * Key for a specific entry in an origin's shared storage.
   * Present only for SharedStorageAccessType.documentSet,
   * SharedStorageAccessType.documentAppend,
   * SharedStorageAccessType.documentDelete,
   * SharedStorageAccessType.workletSet,
   * SharedStorageAccessType.workletAppend,
   * SharedStorageAccessType.workletDelete, and
   * SharedStorageAccessType.workletGet.
   */
  public val key: String? = null,
  /**
   * Value for a specific entry in an origin's shared storage.
   * Present only for SharedStorageAccessType.documentSet,
   * SharedStorageAccessType.documentAppend,
   * SharedStorageAccessType.workletSet, and
   * SharedStorageAccessType.workletAppend.
   */
  public val `value`: String? = null,
  /**
   * Whether or not to set an entry for a key if that key is already present.
   * Present only for SharedStorageAccessType.documentSet and
   * SharedStorageAccessType.workletSet.
   */
  public val ignoreIfPresent: Boolean? = null,
)
