@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.storage.events

import kotlin.String
import kotlin.Suppress
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.network.TimeSinceEpoch
import org.hildan.chrome.devtools.domains.page.FrameId
import org.hildan.chrome.devtools.domains.storage.InterestGroupAccessType
import org.hildan.chrome.devtools.domains.storage.SharedStorageAccessParams
import org.hildan.chrome.devtools.domains.storage.SharedStorageAccessType

@Serializable
public sealed class StorageEvent {
  /**
   * A cache's contents have been modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-cacheStorageContentUpdated)
   */
  @Serializable
  public data class CacheStorageContentUpdated(
    /**
     * Origin to update.
     */
    public val origin: String,
    /**
     * Name of cache in origin.
     */
    public val cacheName: String,
  ) : StorageEvent()

  /**
   * A cache has been added/deleted.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-cacheStorageListUpdated)
   */
  @Serializable
  public data class CacheStorageListUpdated(
    /**
     * Origin to update.
     */
    public val origin: String,
  ) : StorageEvent()

  /**
   * The origin's IndexedDB object store has been modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-indexedDBContentUpdated)
   */
  @Serializable
  public data class IndexedDBContentUpdated(
    /**
     * Origin to update.
     */
    public val origin: String,
    /**
     * Storage key to update.
     */
    public val storageKey: String,
    /**
     * Database to update.
     */
    public val databaseName: String,
    /**
     * ObjectStore to update.
     */
    public val objectStoreName: String,
  ) : StorageEvent()

  /**
   * The origin's IndexedDB database list has been modified.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-indexedDBListUpdated)
   */
  @Serializable
  public data class IndexedDBListUpdated(
    /**
     * Origin to update.
     */
    public val origin: String,
    /**
     * Storage key to update.
     */
    public val storageKey: String,
  ) : StorageEvent()

  /**
   * One of the interest groups was accessed by the associated page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-interestGroupAccessed)
   */
  @Serializable
  public data class InterestGroupAccessed(
    public val accessTime: TimeSinceEpoch,
    public val type: InterestGroupAccessType,
    public val ownerOrigin: String,
    public val name: String,
  ) : StorageEvent()

  /**
   * Shared storage was accessed by the associated page.
   * The following parameters are included in all events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Storage/#event-sharedStorageAccessed)
   */
  @Serializable
  public data class SharedStorageAccessed(
    /**
     * Time of the access.
     */
    public val accessTime: TimeSinceEpoch,
    /**
     * Enum value indicating the Shared Storage API method invoked.
     */
    public val type: SharedStorageAccessType,
    /**
     * DevTools Frame Token for the primary frame tree's root.
     */
    public val mainFrameId: FrameId,
    /**
     * Serialized origin for the context that invoked the Shared Storage API.
     */
    public val ownerOrigin: String,
    /**
     * The sub-parameters warapped by `params` are all optional and their
     * presence/absence depends on `type`.
     */
    public val params: SharedStorageAccessParams,
  ) : StorageEvent()
}
