@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.target

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.browser.BrowserContextID
import org.hildan.chrome.devtools.domains.target.events.TargetEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [TargetDomain.activateTarget] command.
 */
@Serializable
public data class ActivateTargetRequest(
  public val targetId: TargetID,
)

/**
 * A dummy response object for the [TargetDomain.activateTarget] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ActivateTargetResponse

/**
 * Request object containing input parameters for the [TargetDomain.attachToTarget] command.
 */
@Serializable
public data class AttachToTargetRequest(
  public val targetId: TargetID,
  /**
   * Enables "flat" access to the session via specifying sessionId attribute in the commands.
   * We plan to make this the default, deprecate non-flattened mode,
   * and eventually retire it. See crbug.com/991325.
   */
  public val flatten: Boolean? = null,
) {
  public class Builder(
    public val targetId: TargetID,
  ) {
    /**
     * Enables "flat" access to the session via specifying sessionId attribute in the commands.
     * We plan to make this the default, deprecate non-flattened mode,
     * and eventually retire it. See crbug.com/991325.
     */
    public var flatten: Boolean? = null

    public fun build(): AttachToTargetRequest = AttachToTargetRequest(targetId, flatten)
  }
}

/**
 * Response type for the [TargetDomain.attachToTarget] command.
 */
@Serializable
public data class AttachToTargetResponse(
  /**
   * Id assigned to the session.
   */
  public val sessionId: SessionID,
)

/**
 * Response type for the [TargetDomain.attachToBrowserTarget] command.
 */
@Serializable
@ExperimentalChromeApi
public data class AttachToBrowserTargetResponse(
  /**
   * Id assigned to the session.
   */
  public val sessionId: SessionID,
)

/**
 * Request object containing input parameters for the [TargetDomain.closeTarget] command.
 */
@Serializable
public data class CloseTargetRequest(
  public val targetId: TargetID,
)

/**
 * Response type for the [TargetDomain.closeTarget] command.
 */
@Serializable
public data class CloseTargetResponse(
  /**
   * Always set to true. If an error occurs, the response indicates protocol error.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val success: Boolean,
)

/**
 * Request object containing input parameters for the [TargetDomain.exposeDevToolsProtocol] command.
 */
@Serializable
@ExperimentalChromeApi
public data class ExposeDevToolsProtocolRequest(
  public val targetId: TargetID,
  /**
   * Binding name, 'cdp' if not specified.
   */
  public val bindingName: String? = null,
) {
  public class Builder(
    public val targetId: TargetID,
  ) {
    /**
     * Binding name, 'cdp' if not specified.
     */
    public var bindingName: String? = null

    public fun build(): ExposeDevToolsProtocolRequest = ExposeDevToolsProtocolRequest(targetId,
        bindingName)
  }
}

/**
 * A dummy response object for the [TargetDomain.exposeDevToolsProtocol] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object ExposeDevToolsProtocolResponse

/**
 * Request object containing input parameters for the [TargetDomain.createBrowserContext] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CreateBrowserContextRequest(
  /**
   * If specified, disposes this context when debugging session disconnects.
   */
  public val disposeOnDetach: Boolean? = null,
  /**
   * Proxy server, similar to the one passed to --proxy-server
   */
  public val proxyServer: String? = null,
  /**
   * Proxy bypass list, similar to the one passed to --proxy-bypass-list
   */
  public val proxyBypassList: String? = null,
  /**
   * An optional list of origins to grant unlimited cross-origin access to.
   * Parts of the URL other than those constituting origin are ignored.
   */
  public val originsWithUniversalNetworkAccess: List<String>? = null,
) {
  public class Builder() {
    /**
     * If specified, disposes this context when debugging session disconnects.
     */
    public var disposeOnDetach: Boolean? = null

    /**
     * Proxy server, similar to the one passed to --proxy-server
     */
    public var proxyServer: String? = null

    /**
     * Proxy bypass list, similar to the one passed to --proxy-bypass-list
     */
    public var proxyBypassList: String? = null

    /**
     * An optional list of origins to grant unlimited cross-origin access to.
     * Parts of the URL other than those constituting origin are ignored.
     */
    public var originsWithUniversalNetworkAccess: List<String>? = null

    public fun build(): CreateBrowserContextRequest = CreateBrowserContextRequest(disposeOnDetach,
        proxyServer, proxyBypassList, originsWithUniversalNetworkAccess)
  }
}

/**
 * Response type for the [TargetDomain.createBrowserContext] command.
 */
@Serializable
@ExperimentalChromeApi
public data class CreateBrowserContextResponse(
  /**
   * The id of the context created.
   */
  public val browserContextId: BrowserContextID,
)

/**
 * Response type for the [TargetDomain.getBrowserContexts] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetBrowserContextsResponse(
  /**
   * An array of browser context ids.
   */
  public val browserContextIds: List<BrowserContextID>,
)

/**
 * Request object containing input parameters for the [TargetDomain.createTarget] command.
 */
@Serializable
public data class CreateTargetRequest(
  /**
   * The initial URL the page will be navigated to. An empty string indicates about:blank.
   */
  public val url: String,
  /**
   * Frame width in DIP (headless chrome only).
   */
  public val width: Int? = null,
  /**
   * Frame height in DIP (headless chrome only).
   */
  public val height: Int? = null,
  /**
   * The browser context to create the page in.
   */
  @ExperimentalChromeApi
  public val browserContextId: BrowserContextID? = null,
  /**
   * Whether BeginFrames for this target will be controlled via DevTools (headless chrome only,
   * not supported on MacOS yet, false by default).
   */
  @ExperimentalChromeApi
  public val enableBeginFrameControl: Boolean? = null,
  /**
   * Whether to create a new Window or Tab (chrome-only, false by default).
   */
  public val newWindow: Boolean? = null,
  /**
   * Whether to create the target in background or foreground (chrome-only,
   * false by default).
   */
  public val background: Boolean? = null,
) {
  public class Builder(
    /**
     * The initial URL the page will be navigated to. An empty string indicates about:blank.
     */
    public val url: String,
  ) {
    /**
     * Frame width in DIP (headless chrome only).
     */
    public var width: Int? = null

    /**
     * Frame height in DIP (headless chrome only).
     */
    public var height: Int? = null

    /**
     * The browser context to create the page in.
     */
    @ExperimentalChromeApi
    public var browserContextId: BrowserContextID? = null

    /**
     * Whether BeginFrames for this target will be controlled via DevTools (headless chrome only,
     * not supported on MacOS yet, false by default).
     */
    @ExperimentalChromeApi
    public var enableBeginFrameControl: Boolean? = null

    /**
     * Whether to create a new Window or Tab (chrome-only, false by default).
     */
    public var newWindow: Boolean? = null

    /**
     * Whether to create the target in background or foreground (chrome-only,
     * false by default).
     */
    public var background: Boolean? = null

    public fun build(): CreateTargetRequest = CreateTargetRequest(url, width, height,
        browserContextId, enableBeginFrameControl, newWindow, background)
  }
}

/**
 * Response type for the [TargetDomain.createTarget] command.
 */
@Serializable
public data class CreateTargetResponse(
  /**
   * The id of the page opened.
   */
  public val targetId: TargetID,
)

/**
 * Request object containing input parameters for the [TargetDomain.detachFromTarget] command.
 */
@Serializable
public data class DetachFromTargetRequest(
  /**
   * Session to detach.
   */
  public val sessionId: SessionID? = null,
  /**
   * Deprecated.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val targetId: TargetID? = null,
) {
  public class Builder() {
    /**
     * Session to detach.
     */
    public var sessionId: SessionID? = null

    /**
     * Deprecated.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var targetId: TargetID? = null

    public fun build(): DetachFromTargetRequest = DetachFromTargetRequest(sessionId, targetId)
  }
}

/**
 * A dummy response object for the [TargetDomain.detachFromTarget] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object DetachFromTargetResponse

/**
 * Request object containing input parameters for the [TargetDomain.disposeBrowserContext] command.
 */
@Serializable
@ExperimentalChromeApi
public data class DisposeBrowserContextRequest(
  public val browserContextId: BrowserContextID,
)

/**
 * A dummy response object for the [TargetDomain.disposeBrowserContext] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object DisposeBrowserContextResponse

/**
 * Request object containing input parameters for the [TargetDomain.getTargetInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetTargetInfoRequest(
  public val targetId: TargetID? = null,
) {
  public class Builder() {
    public var targetId: TargetID? = null

    public fun build(): GetTargetInfoRequest = GetTargetInfoRequest(targetId)
  }
}

/**
 * Response type for the [TargetDomain.getTargetInfo] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetTargetInfoResponse(
  public val targetInfo: TargetInfo,
)

/**
 * Request object containing input parameters for the [TargetDomain.getTargets] command.
 */
@Serializable
public data class GetTargetsRequest(
  /**
   * Only targets matching filter will be reported. If filter is not specified
   * and target discovery is currently enabled, a filter used for target discovery
   * is used for consistency.
   */
  @ExperimentalChromeApi
  public val filter: TargetFilter? = null,
) {
  public class Builder() {
    /**
     * Only targets matching filter will be reported. If filter is not specified
     * and target discovery is currently enabled, a filter used for target discovery
     * is used for consistency.
     */
    @ExperimentalChromeApi
    public var filter: TargetFilter? = null

    public fun build(): GetTargetsRequest = GetTargetsRequest(filter)
  }
}

/**
 * Response type for the [TargetDomain.getTargets] command.
 */
@Serializable
public data class GetTargetsResponse(
  /**
   * The list of targets.
   */
  public val targetInfos: List<TargetInfo>,
)

/**
 * Request object containing input parameters for the [TargetDomain.sendMessageToTarget] command.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public data class SendMessageToTargetRequest(
  public val message: String,
  /**
   * Identifier of the session.
   */
  public val sessionId: SessionID? = null,
  /**
   * Deprecated.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val targetId: TargetID? = null,
) {
  public class Builder(
    public val message: String,
  ) {
    /**
     * Identifier of the session.
     */
    public var sessionId: SessionID? = null

    /**
     * Deprecated.
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var targetId: TargetID? = null

    public fun build(): SendMessageToTargetRequest = SendMessageToTargetRequest(message, sessionId,
        targetId)
  }
}

/**
 * A dummy response object for the [TargetDomain.sendMessageToTarget] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@Deprecated(message = "Deprecated in the Chrome DevTools protocol")
public object SendMessageToTargetResponse

/**
 * Request object containing input parameters for the [TargetDomain.setAutoAttach] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetAutoAttachRequest(
  /**
   * Whether to auto-attach to related targets.
   */
  public val autoAttach: Boolean,
  /**
   * Whether to pause new targets when attaching to them. Use `Runtime.runIfWaitingForDebugger`
   * to run paused targets.
   */
  public val waitForDebuggerOnStart: Boolean,
  /**
   * Enables "flat" access to the session via specifying sessionId attribute in the commands.
   * We plan to make this the default, deprecate non-flattened mode,
   * and eventually retire it. See crbug.com/991325.
   */
  public val flatten: Boolean? = null,
  /**
   * Only targets matching filter will be attached.
   */
  @ExperimentalChromeApi
  public val filter: TargetFilter? = null,
) {
  public class Builder(
    /**
     * Whether to auto-attach to related targets.
     */
    public val autoAttach: Boolean,
    /**
     * Whether to pause new targets when attaching to them. Use `Runtime.runIfWaitingForDebugger`
     * to run paused targets.
     */
    public val waitForDebuggerOnStart: Boolean,
  ) {
    /**
     * Enables "flat" access to the session via specifying sessionId attribute in the commands.
     * We plan to make this the default, deprecate non-flattened mode,
     * and eventually retire it. See crbug.com/991325.
     */
    public var flatten: Boolean? = null

    /**
     * Only targets matching filter will be attached.
     */
    @ExperimentalChromeApi
    public var filter: TargetFilter? = null

    public fun build(): SetAutoAttachRequest = SetAutoAttachRequest(autoAttach,
        waitForDebuggerOnStart, flatten, filter)
  }
}

/**
 * A dummy response object for the [TargetDomain.setAutoAttach] command. This command doesn't return
 * any result at the moment, but this could happen in the future, or could have happened in the past.
 * For forwards and backwards compatibility of the command method, we still declare this class even
 * without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetAutoAttachResponse

/**
 * Request object containing input parameters for the [TargetDomain.autoAttachRelated] command.
 */
@Serializable
@ExperimentalChromeApi
public data class AutoAttachRelatedRequest(
  public val targetId: TargetID,
  /**
   * Whether to pause new targets when attaching to them. Use `Runtime.runIfWaitingForDebugger`
   * to run paused targets.
   */
  public val waitForDebuggerOnStart: Boolean,
  /**
   * Only targets matching filter will be attached.
   */
  @ExperimentalChromeApi
  public val filter: TargetFilter? = null,
) {
  public class Builder(
    public val targetId: TargetID,
    /**
     * Whether to pause new targets when attaching to them. Use `Runtime.runIfWaitingForDebugger`
     * to run paused targets.
     */
    public val waitForDebuggerOnStart: Boolean,
  ) {
    /**
     * Only targets matching filter will be attached.
     */
    @ExperimentalChromeApi
    public var filter: TargetFilter? = null

    public fun build(): AutoAttachRelatedRequest = AutoAttachRelatedRequest(targetId,
        waitForDebuggerOnStart, filter)
  }
}

/**
 * A dummy response object for the [TargetDomain.autoAttachRelated] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object AutoAttachRelatedResponse

/**
 * Request object containing input parameters for the [TargetDomain.setDiscoverTargets] command.
 */
@Serializable
public data class SetDiscoverTargetsRequest(
  /**
   * Whether to discover available targets.
   */
  public val discover: Boolean,
  /**
   * Only targets matching filter will be attached. If `discover` is false,
   * `filter` must be omitted or empty.
   */
  @ExperimentalChromeApi
  public val filter: TargetFilter? = null,
) {
  public class Builder(
    /**
     * Whether to discover available targets.
     */
    public val discover: Boolean,
  ) {
    /**
     * Only targets matching filter will be attached. If `discover` is false,
     * `filter` must be omitted or empty.
     */
    @ExperimentalChromeApi
    public var filter: TargetFilter? = null

    public fun build(): SetDiscoverTargetsRequest = SetDiscoverTargetsRequest(discover, filter)
  }
}

/**
 * A dummy response object for the [TargetDomain.setDiscoverTargets] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetDiscoverTargetsResponse

/**
 * Request object containing input parameters for the [TargetDomain.setRemoteLocations] command.
 */
@Serializable
@ExperimentalChromeApi
public data class SetRemoteLocationsRequest(
  /**
   * List of remote locations.
   */
  public val locations: List<RemoteLocation>,
)

/**
 * A dummy response object for the [TargetDomain.setRemoteLocations] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
@ExperimentalChromeApi
public object SetRemoteLocationsResponse

/**
 * Supports additional targets discovery and allows to attach to them.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target)
 */
public class TargetDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out TargetEvent>> =
      mapOf(
      "Target.attachedToTarget" to serializer<TargetEvent.AttachedToTarget>(),
      "Target.detachedFromTarget" to serializer<TargetEvent.DetachedFromTarget>(),
      "Target.receivedMessageFromTarget" to serializer<TargetEvent.ReceivedMessageFromTarget>(),
      "Target.targetCreated" to serializer<TargetEvent.TargetCreated>(),
      "Target.targetDestroyed" to serializer<TargetEvent.TargetDestroyed>(),
      "Target.targetCrashed" to serializer<TargetEvent.TargetCrashed>(),
      "Target.targetInfoChanged" to serializer<TargetEvent.TargetInfoChanged>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<TargetEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Issued when attached to target because of auto-attach or `attachToTarget` command.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-attachedToTarget)
   */
  @ExperimentalChromeApi
  public fun attachedToTargetEvents(): Flow<TargetEvent.AttachedToTarget> =
      session.typedEvents("Target.attachedToTarget")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("attachedToTargetEvents()"),
  )
  public fun attachedToTarget(): Flow<TargetEvent.AttachedToTarget> = attachedToTargetEvents()

  /**
   * Issued when detached from target for any reason (including `detachFromTarget` command). Can be
   * issued multiple times per target if multiple sessions have been attached to it.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-detachedFromTarget)
   */
  @ExperimentalChromeApi
  public fun detachedFromTargetEvents(): Flow<TargetEvent.DetachedFromTarget> =
      session.typedEvents("Target.detachedFromTarget")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("detachedFromTargetEvents()"),
  )
  public fun detachedFromTarget(): Flow<TargetEvent.DetachedFromTarget> = detachedFromTargetEvents()

  /**
   * Notifies about a new protocol message received from the session (as reported in
   * `attachedToTarget` event).
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-receivedMessageFromTarget)
   */
  public fun receivedMessageFromTargetEvents(): Flow<TargetEvent.ReceivedMessageFromTarget> =
      session.typedEvents("Target.receivedMessageFromTarget")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("receivedMessageFromTargetEvents()"),
  )
  public fun receivedMessageFromTarget(): Flow<TargetEvent.ReceivedMessageFromTarget> =
      receivedMessageFromTargetEvents()

  /**
   * Issued when a possible inspection target is created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetCreated)
   */
  public fun targetCreatedEvents(): Flow<TargetEvent.TargetCreated> =
      session.typedEvents("Target.targetCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("targetCreatedEvents()"),
  )
  public fun targetCreated(): Flow<TargetEvent.TargetCreated> = targetCreatedEvents()

  /**
   * Issued when a target is destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetDestroyed)
   */
  public fun targetDestroyedEvents(): Flow<TargetEvent.TargetDestroyed> =
      session.typedEvents("Target.targetDestroyed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("targetDestroyedEvents()"),
  )
  public fun targetDestroyed(): Flow<TargetEvent.TargetDestroyed> = targetDestroyedEvents()

  /**
   * Issued when a target has crashed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetCrashed)
   */
  public fun targetCrashedEvents(): Flow<TargetEvent.TargetCrashed> =
      session.typedEvents("Target.targetCrashed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("targetCrashedEvents()"),
  )
  public fun targetCrashed(): Flow<TargetEvent.TargetCrashed> = targetCrashedEvents()

  /**
   * Issued when some information about a target has changed. This only happens between
   * `targetCreated` and `targetDestroyed`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#event-targetInfoChanged)
   */
  public fun targetInfoChangedEvents(): Flow<TargetEvent.TargetInfoChanged> =
      session.typedEvents("Target.targetInfoChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("targetInfoChangedEvents()"),
  )
  public fun targetInfoChanged(): Flow<TargetEvent.TargetInfoChanged> = targetInfoChangedEvents()

  /**
   * Activates (focuses) the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-activateTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun activateTarget(input: ActivateTargetRequest): ActivateTargetResponse =
      session.request("Target.activateTarget", input)

  /**
   * Activates (focuses) the target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-activateTarget)
   */
  public suspend fun activateTarget(targetId: TargetID): ActivateTargetResponse {
    val input = ActivateTargetRequest(targetId)
    return activateTarget(input)
  }

  /**
   * Attaches to the target with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-attachToTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun attachToTarget(input: AttachToTargetRequest): AttachToTargetResponse =
      session.request("Target.attachToTarget", input)

  /**
   * Attaches to the target with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-attachToTarget)
   */
  @JvmOverloads
  public suspend inline fun attachToTarget(targetId: TargetID,
      optionalArgs: AttachToTargetRequest.Builder.() -> Unit = {}): AttachToTargetResponse {
    val builder = AttachToTargetRequest.Builder(targetId)
    val input = builder.apply(optionalArgs).build()
    return attachToTarget(input)
  }

  /**
   * Attaches to the browser target, only uses flat sessionId mode.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-attachToBrowserTarget)
   */
  @ExperimentalChromeApi
  public suspend fun attachToBrowserTarget(): AttachToBrowserTargetResponse =
      session.request("Target.attachToBrowserTarget", Unit)

  /**
   * Closes the target. If the target is a page that gets closed too.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-closeTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun closeTarget(input: CloseTargetRequest): CloseTargetResponse =
      session.request("Target.closeTarget", input)

  /**
   * Closes the target. If the target is a page that gets closed too.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-closeTarget)
   */
  public suspend fun closeTarget(targetId: TargetID): CloseTargetResponse {
    val input = CloseTargetRequest(targetId)
    return closeTarget(input)
  }

  /**
   * Inject object to the target's main frame that provides a communication
   * channel with browser target.
   *
   * Injected object will be available as `window[bindingName]`.
   *
   * The object has the follwing API:
   * - `binding.send(json)` - a method to send messages over the remote debugging protocol
   * - `binding.onmessage = json => handleMessage(json)` - a callback that will be called for the
   * protocol notifications and command responses.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-exposeDevToolsProtocol)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun exposeDevToolsProtocol(input: ExposeDevToolsProtocolRequest):
      ExposeDevToolsProtocolResponse = session.request("Target.exposeDevToolsProtocol", input)

  /**
   * Inject object to the target's main frame that provides a communication
   * channel with browser target.
   *
   * Injected object will be available as `window[bindingName]`.
   *
   * The object has the follwing API:
   * - `binding.send(json)` - a method to send messages over the remote debugging protocol
   * - `binding.onmessage = json => handleMessage(json)` - a callback that will be called for the
   * protocol notifications and command responses.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-exposeDevToolsProtocol)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun exposeDevToolsProtocol(targetId: TargetID,
      optionalArgs: ExposeDevToolsProtocolRequest.Builder.() -> Unit = {}):
      ExposeDevToolsProtocolResponse {
    val builder = ExposeDevToolsProtocolRequest.Builder(targetId)
    val input = builder.apply(optionalArgs).build()
    return exposeDevToolsProtocol(input)
  }

  /**
   * Creates a new empty BrowserContext. Similar to an incognito profile but you can have more than
   * one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-createBrowserContext)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun createBrowserContext(input: CreateBrowserContextRequest):
      CreateBrowserContextResponse = session.request("Target.createBrowserContext", input)

  /**
   * Creates a new empty BrowserContext. Similar to an incognito profile but you can have more than
   * one.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-createBrowserContext)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline
      fun createBrowserContext(optionalArgs: CreateBrowserContextRequest.Builder.() -> Unit = {}):
      CreateBrowserContextResponse {
    val builder = CreateBrowserContextRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return createBrowserContext(input)
  }

  /**
   * Returns all browser contexts created with `Target.createBrowserContext` method.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getBrowserContexts)
   */
  @ExperimentalChromeApi
  public suspend fun getBrowserContexts(): GetBrowserContextsResponse =
      session.request("Target.getBrowserContexts", Unit)

  /**
   * Creates a new page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-createTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun createTarget(input: CreateTargetRequest): CreateTargetResponse =
      session.request("Target.createTarget", input)

  /**
   * Creates a new page.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-createTarget)
   *
   * @param url The initial URL the page will be navigated to. An empty string indicates
   * about:blank.
   */
  @JvmOverloads
  public suspend inline fun createTarget(url: String,
      optionalArgs: CreateTargetRequest.Builder.() -> Unit = {}): CreateTargetResponse {
    val builder = CreateTargetRequest.Builder(url)
    val input = builder.apply(optionalArgs).build()
    return createTarget(input)
  }

  /**
   * Detaches session with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-detachFromTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun detachFromTarget(input: DetachFromTargetRequest): DetachFromTargetResponse =
      session.request("Target.detachFromTarget", input)

  /**
   * Detaches session with given id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-detachFromTarget)
   */
  @JvmOverloads
  public suspend inline
      fun detachFromTarget(optionalArgs: DetachFromTargetRequest.Builder.() -> Unit = {}):
      DetachFromTargetResponse {
    val builder = DetachFromTargetRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return detachFromTarget(input)
  }

  /**
   * Deletes a BrowserContext. All the belonging pages will be closed without calling their
   * beforeunload hooks.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-disposeBrowserContext)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun disposeBrowserContext(input: DisposeBrowserContextRequest):
      DisposeBrowserContextResponse = session.request("Target.disposeBrowserContext", input)

  /**
   * Deletes a BrowserContext. All the belonging pages will be closed without calling their
   * beforeunload hooks.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-disposeBrowserContext)
   */
  @ExperimentalChromeApi
  public suspend fun disposeBrowserContext(browserContextId: BrowserContextID):
      DisposeBrowserContextResponse {
    val input = DisposeBrowserContextRequest(browserContextId)
    return disposeBrowserContext(input)
  }

  /**
   * Returns information about a target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getTargetInfo)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun getTargetInfo(input: GetTargetInfoRequest): GetTargetInfoResponse =
      session.request("Target.getTargetInfo", input)

  /**
   * Returns information about a target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getTargetInfo)
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun getTargetInfo(optionalArgs: GetTargetInfoRequest.Builder.() -> Unit =
      {}): GetTargetInfoResponse {
    val builder = GetTargetInfoRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getTargetInfo(input)
  }

  /**
   * Retrieves a list of available targets.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getTargets)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getTargets(input: GetTargetsRequest): GetTargetsResponse =
      session.request("Target.getTargets", input)

  /**
   * Retrieves a list of available targets.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-getTargets)
   */
  @JvmOverloads
  public suspend inline fun getTargets(optionalArgs: GetTargetsRequest.Builder.() -> Unit = {}):
      GetTargetsResponse {
    val builder = GetTargetsRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return getTargets(input)
  }

  /**
   * Sends protocol message over session with given id.
   * Consider using flat mode instead; see commands attachToTarget, setAutoAttach,
   * and crbug.com/991325.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-sendMessageToTarget)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public suspend fun sendMessageToTarget(input: SendMessageToTargetRequest):
      SendMessageToTargetResponse = session.request("Target.sendMessageToTarget", input)

  /**
   * Sends protocol message over session with given id.
   * Consider using flat mode instead; see commands attachToTarget, setAutoAttach,
   * and crbug.com/991325.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-sendMessageToTarget)
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  @JvmOverloads
  public suspend inline fun sendMessageToTarget(message: String,
      optionalArgs: SendMessageToTargetRequest.Builder.() -> Unit = {}):
      SendMessageToTargetResponse {
    val builder = SendMessageToTargetRequest.Builder(message)
    val input = builder.apply(optionalArgs).build()
    return sendMessageToTarget(input)
  }

  /**
   * Controls whether to automatically attach to new targets which are considered to be related to
   * this one. When turned on, attaches to all existing related targets as well. When turned off,
   * automatically detaches from all currently attached targets.
   * This also clears all targets added by `autoAttachRelated` from the list of targets to watch
   * for creation of related targets.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setAutoAttach)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setAutoAttach(input: SetAutoAttachRequest): SetAutoAttachResponse =
      session.request("Target.setAutoAttach", input)

  /**
   * Controls whether to automatically attach to new targets which are considered to be related to
   * this one. When turned on, attaches to all existing related targets as well. When turned off,
   * automatically detaches from all currently attached targets.
   * This also clears all targets added by `autoAttachRelated` from the list of targets to watch
   * for creation of related targets.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setAutoAttach)
   *
   * @param autoAttach Whether to auto-attach to related targets.
   * @param waitForDebuggerOnStart Whether to pause new targets when attaching to them. Use
   * `Runtime.runIfWaitingForDebugger`
   * to run paused targets.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun setAutoAttach(
    autoAttach: Boolean,
    waitForDebuggerOnStart: Boolean,
    optionalArgs: SetAutoAttachRequest.Builder.() -> Unit = {},
  ): SetAutoAttachResponse {
    val builder = SetAutoAttachRequest.Builder(autoAttach, waitForDebuggerOnStart)
    val input = builder.apply(optionalArgs).build()
    return setAutoAttach(input)
  }

  /**
   * Adds the specified target to the list of targets that will be monitored for any related target
   * creation (such as child frames, child workers and new versions of service worker) and reported
   * through `attachedToTarget`. The specified target is also auto-attached.
   * This cancels the effect of any previous `setAutoAttach` and is also cancelled by subsequent
   * `setAutoAttach`. Only available at the Browser target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-autoAttachRelated)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun autoAttachRelated(input: AutoAttachRelatedRequest): AutoAttachRelatedResponse =
      session.request("Target.autoAttachRelated", input)

  /**
   * Adds the specified target to the list of targets that will be monitored for any related target
   * creation (such as child frames, child workers and new versions of service worker) and reported
   * through `attachedToTarget`. The specified target is also auto-attached.
   * This cancels the effect of any previous `setAutoAttach` and is also cancelled by subsequent
   * `setAutoAttach`. Only available at the Browser target.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-autoAttachRelated)
   * @param waitForDebuggerOnStart Whether to pause new targets when attaching to them. Use
   * `Runtime.runIfWaitingForDebugger`
   * to run paused targets.
   */
  @ExperimentalChromeApi
  @JvmOverloads
  public suspend inline fun autoAttachRelated(
    targetId: TargetID,
    waitForDebuggerOnStart: Boolean,
    optionalArgs: AutoAttachRelatedRequest.Builder.() -> Unit = {},
  ): AutoAttachRelatedResponse {
    val builder = AutoAttachRelatedRequest.Builder(targetId, waitForDebuggerOnStart)
    val input = builder.apply(optionalArgs).build()
    return autoAttachRelated(input)
  }

  /**
   * Controls whether to discover available targets and notify via
   * `targetCreated/targetInfoChanged/targetDestroyed` events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setDiscoverTargets)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setDiscoverTargets(input: SetDiscoverTargetsRequest):
      SetDiscoverTargetsResponse = session.request("Target.setDiscoverTargets", input)

  /**
   * Controls whether to discover available targets and notify via
   * `targetCreated/targetInfoChanged/targetDestroyed` events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setDiscoverTargets)
   *
   * @param discover Whether to discover available targets.
   */
  @JvmOverloads
  public suspend inline fun setDiscoverTargets(discover: Boolean,
      optionalArgs: SetDiscoverTargetsRequest.Builder.() -> Unit = {}): SetDiscoverTargetsResponse {
    val builder = SetDiscoverTargetsRequest.Builder(discover)
    val input = builder.apply(optionalArgs).build()
    return setDiscoverTargets(input)
  }

  /**
   * Enables target discovery for the specified locations, when `setDiscoverTargets` was set to
   * `true`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setRemoteLocations)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  @ExperimentalChromeApi
  public suspend fun setRemoteLocations(input: SetRemoteLocationsRequest):
      SetRemoteLocationsResponse = session.request("Target.setRemoteLocations", input)

  /**
   * Enables target discovery for the specified locations, when `setDiscoverTargets` was set to
   * `true`.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Target/#method-setRemoteLocations)
   *
   * @param locations List of remote locations.
   */
  @ExperimentalChromeApi
  public suspend fun setRemoteLocations(locations: List<RemoteLocation>):
      SetRemoteLocationsResponse {
    val input = SetRemoteLocationsRequest(locations)
    return setRemoteLocations(input)
  }
}
