@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.tethering

import kotlin.Deprecated
import kotlin.Int
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.tethering.events.TetheringEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * Request object containing input parameters for the [TetheringDomain.bind] command.
 */
@Serializable
public data class BindRequest(
  /**
   * Port number to bind.
   */
  public val port: Int,
)

/**
 * A dummy response object for the [TetheringDomain.bind] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object BindResponse

/**
 * Request object containing input parameters for the [TetheringDomain.unbind] command.
 */
@Serializable
public data class UnbindRequest(
  /**
   * Port number to unbind.
   */
  public val port: Int,
)

/**
 * A dummy response object for the [TetheringDomain.unbind] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object UnbindResponse

/**
 * The Tethering domain defines methods and events for browser port binding.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering)
 */
@ExperimentalChromeApi
public class TetheringDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out TetheringEvent>> =
      mapOf(
      "Tethering.accepted" to serializer<TetheringEvent.Accepted>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<TetheringEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Informs that port was successfully bound and got a specified connection id.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#event-accepted)
   */
  public fun acceptedEvents(): Flow<TetheringEvent.Accepted> =
      session.typedEvents("Tethering.accepted")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("acceptedEvents()"),
  )
  public fun accepted(): Flow<TetheringEvent.Accepted> = acceptedEvents()

  /**
   * Request browser port binding.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#method-bind)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun bind(input: BindRequest): BindResponse = session.request("Tethering.bind",
      input)

  /**
   * Request browser port binding.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#method-bind)
   *
   * @param port Port number to bind.
   */
  public suspend fun bind(port: Int): BindResponse {
    val input = BindRequest(port)
    return bind(input)
  }

  /**
   * Request browser port unbinding.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#method-unbind)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun unbind(input: UnbindRequest): UnbindResponse =
      session.request("Tethering.unbind", input)

  /**
   * Request browser port unbinding.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tethering/#method-unbind)
   *
   * @param port Port number to unbind.
   */
  public suspend fun unbind(port: Int): UnbindResponse {
    val input = UnbindRequest(port)
    return unbind(input)
  }
}
