@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.tracing

import kotlin.Boolean
import kotlin.Deprecated
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.tracing.events.TracingEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [TracingDomain.end] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EndResponse

/**
 * Response type for the [TracingDomain.getCategories] command.
 */
@Serializable
public data class GetCategoriesResponse(
  /**
   * A list of supported tracing categories.
   */
  public val categories: List<String>,
)

/**
 * Request object containing input parameters for the [TracingDomain.recordClockSyncMarker] command.
 */
@Serializable
public data class RecordClockSyncMarkerRequest(
  /**
   * The ID of this clock sync marker
   */
  public val syncId: String,
)

/**
 * A dummy response object for the [TracingDomain.recordClockSyncMarker] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object RecordClockSyncMarkerResponse

/**
 * Request object containing input parameters for the [TracingDomain.requestMemoryDump] command.
 */
@Serializable
public data class RequestMemoryDumpRequest(
  /**
   * Enables more deterministic results by forcing garbage collection
   */
  public val deterministic: Boolean? = null,
  /**
   * Specifies level of details in memory dump. Defaults to "detailed".
   */
  public val levelOfDetail: MemoryDumpLevelOfDetail? = null,
) {
  public class Builder() {
    /**
     * Enables more deterministic results by forcing garbage collection
     */
    public var deterministic: Boolean? = null

    /**
     * Specifies level of details in memory dump. Defaults to "detailed".
     */
    public var levelOfDetail: MemoryDumpLevelOfDetail? = null

    public fun build(): RequestMemoryDumpRequest = RequestMemoryDumpRequest(deterministic,
        levelOfDetail)
  }
}

/**
 * Response type for the [TracingDomain.requestMemoryDump] command.
 */
@Serializable
public data class RequestMemoryDumpResponse(
  /**
   * GUID of the resulting global memory dump.
   */
  public val dumpGuid: String,
  /**
   * True iff the global memory dump succeeded.
   */
  public val success: Boolean,
)

/**
 * Request object containing input parameters for the [TracingDomain.start] command.
 */
@Serializable
public data class StartRequest(
  /**
   * Category/tag filter
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val categories: String? = null,
  /**
   * Tracing options
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val options: String? = null,
  /**
   * If set, the agent will issue bufferUsage events at this interval, specified in milliseconds
   */
  public val bufferUsageReportingInterval: Double? = null,
  /**
   * Whether to report trace events as series of dataCollected events or to save trace to a
   * stream (defaults to `ReportEvents`).
   */
  public val transferMode: TransferMode? = null,
  /**
   * Trace data format to use. This only applies when using `ReturnAsStream`
   * transfer mode (defaults to `json`).
   */
  public val streamFormat: StreamFormat? = null,
  /**
   * Compression format to use. This only applies when using `ReturnAsStream`
   * transfer mode (defaults to `none`)
   */
  public val streamCompression: StreamCompression? = null,
  public val traceConfig: TraceConfig? = null,
  /**
   * Base64-encoded serialized perfetto.protos.TraceConfig protobuf message
   * When specified, the parameters `categories`, `options`, `traceConfig`
   * are ignored. (Encoded as a base64 string when passed over JSON)
   */
  public val perfettoConfig: String? = null,
  /**
   * Backend type (defaults to `auto`)
   */
  public val tracingBackend: TracingBackend? = null,
) {
  public class Builder() {
    /**
     * Category/tag filter
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var categories: String? = null

    /**
     * Tracing options
     */
    @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
    public var options: String? = null

    /**
     * If set, the agent will issue bufferUsage events at this interval, specified in milliseconds
     */
    public var bufferUsageReportingInterval: Double? = null

    /**
     * Whether to report trace events as series of dataCollected events or to save trace to a
     * stream (defaults to `ReportEvents`).
     */
    public var transferMode: TransferMode? = null

    /**
     * Trace data format to use. This only applies when using `ReturnAsStream`
     * transfer mode (defaults to `json`).
     */
    public var streamFormat: StreamFormat? = null

    /**
     * Compression format to use. This only applies when using `ReturnAsStream`
     * transfer mode (defaults to `none`)
     */
    public var streamCompression: StreamCompression? = null

    public var traceConfig: TraceConfig? = null

    /**
     * Base64-encoded serialized perfetto.protos.TraceConfig protobuf message
     * When specified, the parameters `categories`, `options`, `traceConfig`
     * are ignored. (Encoded as a base64 string when passed over JSON)
     */
    public var perfettoConfig: String? = null

    /**
     * Backend type (defaults to `auto`)
     */
    public var tracingBackend: TracingBackend? = null

    public fun build(): StartRequest = StartRequest(categories, options,
        bufferUsageReportingInterval, transferMode, streamFormat, streamCompression, traceConfig,
        perfettoConfig, tracingBackend)
  }
}

/**
 * A dummy response object for the [TracingDomain.start] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object StartResponse

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing)
 */
@ExperimentalChromeApi
public class TracingDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out TracingEvent>> =
      mapOf(
      "Tracing.bufferUsage" to serializer<TracingEvent.BufferUsage>(),
      "Tracing.dataCollected" to serializer<TracingEvent.DataCollected>(),
      "Tracing.tracingComplete" to serializer<TracingEvent.TracingComplete>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<TracingEvent> = session.typedEvents(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#event-bufferUsage)
   */
  public fun bufferUsageEvents(): Flow<TracingEvent.BufferUsage> =
      session.typedEvents("Tracing.bufferUsage")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("bufferUsageEvents()"),
  )
  public fun bufferUsage(): Flow<TracingEvent.BufferUsage> = bufferUsageEvents()

  /**
   * Contains an bucket of collected trace events. When tracing is stopped collected events will be
   * send as a sequence of dataCollected events followed by tracingComplete event.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#event-dataCollected)
   */
  public fun dataCollectedEvents(): Flow<TracingEvent.DataCollected> =
      session.typedEvents("Tracing.dataCollected")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("dataCollectedEvents()"),
  )
  public fun dataCollected(): Flow<TracingEvent.DataCollected> = dataCollectedEvents()

  /**
   * Signals that tracing is stopped and there is no trace buffers pending flush, all data were
   * delivered via dataCollected events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#event-tracingComplete)
   */
  public fun tracingCompleteEvents(): Flow<TracingEvent.TracingComplete> =
      session.typedEvents("Tracing.tracingComplete")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("tracingCompleteEvents()"),
  )
  public fun tracingComplete(): Flow<TracingEvent.TracingComplete> = tracingCompleteEvents()

  /**
   * Stop trace events collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-end)
   */
  public suspend fun end(): EndResponse = session.request("Tracing.end", Unit)

  /**
   * Gets supported tracing categories.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-getCategories)
   */
  public suspend fun getCategories(): GetCategoriesResponse =
      session.request("Tracing.getCategories", Unit)

  /**
   * Record a clock sync marker in the trace.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-recordClockSyncMarker)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun recordClockSyncMarker(input: RecordClockSyncMarkerRequest):
      RecordClockSyncMarkerResponse = session.request("Tracing.recordClockSyncMarker", input)

  /**
   * Record a clock sync marker in the trace.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-recordClockSyncMarker)
   *
   * @param syncId The ID of this clock sync marker
   */
  public suspend fun recordClockSyncMarker(syncId: String): RecordClockSyncMarkerResponse {
    val input = RecordClockSyncMarkerRequest(syncId)
    return recordClockSyncMarker(input)
  }

  /**
   * Request a global memory dump.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-requestMemoryDump)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun requestMemoryDump(input: RequestMemoryDumpRequest): RequestMemoryDumpResponse =
      session.request("Tracing.requestMemoryDump", input)

  /**
   * Request a global memory dump.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-requestMemoryDump)
   */
  @JvmOverloads
  public suspend inline
      fun requestMemoryDump(optionalArgs: RequestMemoryDumpRequest.Builder.() -> Unit = {}):
      RequestMemoryDumpResponse {
    val builder = RequestMemoryDumpRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return requestMemoryDump(input)
  }

  /**
   * Start trace events collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-start)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun start(input: StartRequest): StartResponse = session.request("Tracing.start",
      input)

  /**
   * Start trace events collection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#method-start)
   */
  @JvmOverloads
  public suspend inline fun start(optionalArgs: StartRequest.Builder.() -> Unit = {}):
      StartResponse {
    val builder = StartRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return start(input)
  }
}
