@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.tracing

import kotlin.Boolean
import kotlin.Double
import kotlin.String
import kotlin.Suppress
import kotlin.collections.List
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject

/**
 * Configuration for memory dump. Used only when "memory-infra" category is enabled.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-MemoryDumpConfig)
 */
public typealias MemoryDumpConfig = JsonObject

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-TraceConfig)
 */
@Serializable
public data class TraceConfig(
  /**
   * Controls how the trace buffer stores data.
   */
  public val recordMode: TraceConfigRecordMode? = null,
  /**
   * Size of the trace buffer in kilobytes. If not specified or zero is passed, a default value
   * of 200 MB would be used.
   */
  public val traceBufferSizeInKb: Double? = null,
  /**
   * Turns on JavaScript stack sampling.
   */
  public val enableSampling: Boolean? = null,
  /**
   * Turns on system tracing.
   */
  public val enableSystrace: Boolean? = null,
  /**
   * Turns on argument filter.
   */
  public val enableArgumentFilter: Boolean? = null,
  /**
   * Included category filters.
   */
  public val includedCategories: List<String>? = null,
  /**
   * Excluded category filters.
   */
  public val excludedCategories: List<String>? = null,
  /**
   * Configuration to synthesize the delays in tracing.
   */
  public val syntheticDelays: List<String>? = null,
  /**
   * Configuration for memory dump triggers. Used only when "memory-infra" category is enabled.
   */
  public val memoryDumpConfig: MemoryDumpConfig? = null,
)

/**
 * Data format of a trace. Can be either the legacy JSON format or the
 * protocol buffer format. Note that the JSON format will be deprecated soon.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-StreamFormat)
 */
@Serializable
public enum class StreamFormat {
  json,
  proto,
}

/**
 * Compression type to use for traces returned via streams.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-StreamCompression)
 */
@Serializable
public enum class StreamCompression {
  none,
  gzip,
}

/**
 * Details exposed when memory request explicitly declared.
 * Keep consistent with memory_dump_request_args.h and
 * memory_instrumentation.mojom
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-MemoryDumpLevelOfDetail)
 */
@Serializable
public enum class MemoryDumpLevelOfDetail {
  background,
  light,
  detailed,
}

/**
 * Backend type to use for tracing. `chrome` uses the Chrome-integrated
 * tracing service and is supported on all platforms. `system` is only
 * supported on Chrome OS and uses the Perfetto system tracing service.
 * `auto` chooses `system` when the perfettoConfig provided to Tracing.start
 * specifies at least one non-Chrome data source; otherwise uses `chrome`.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-TracingBackend)
 */
@Serializable
public enum class TracingBackend {
  auto,
  chrome,
  system,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the property 'recordMode' of the type 'TraceConfig'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-TraceConfigRecordMode)
 */
@Serializable
public enum class TraceConfigRecordMode {
  recordUntilFull,
  recordContinuously,
  recordAsMuchAsPossible,
  echoToConsole,
}

/**
 * This enum doesn't have a proper description because it was generated from an inline declaration.
 * Its name was inferred based on the place where it is used:
 *  - the parameter 'transferMode' of the command 'Tracing.start'
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Tracing/#type-TransferMode)
 */
@Serializable
public enum class TransferMode {
  ReportEvents,
  ReturnAsStream,
}
