@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.webaudio

import kotlin.Deprecated
import kotlin.String
import kotlin.Suppress
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.webaudio.events.WebAudioEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request
import org.hildan.chrome.devtools.protocol.typedEvents

/**
 * A dummy response object for the [WebAudioDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * A dummy response object for the [WebAudioDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [WebAudioDomain.getRealtimeData] command.
 */
@Serializable
public data class GetRealtimeDataRequest(
  public val contextId: GraphObjectId,
)

/**
 * Response type for the [WebAudioDomain.getRealtimeData] command.
 */
@Serializable
public data class GetRealtimeDataResponse(
  public val realtimeData: ContextRealtimeData,
)

/**
 * This domain allows inspection of Web Audio API.
 * https://webaudio.github.io/web-audio-api/
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio)
 */
@ExperimentalChromeApi
public class WebAudioDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out WebAudioEvent>> =
      mapOf(
      "WebAudio.contextCreated" to serializer<WebAudioEvent.ContextCreated>(),
      "WebAudio.contextWillBeDestroyed" to serializer<WebAudioEvent.ContextWillBeDestroyed>(),
      "WebAudio.contextChanged" to serializer<WebAudioEvent.ContextChanged>(),
      "WebAudio.audioListenerCreated" to serializer<WebAudioEvent.AudioListenerCreated>(),
      "WebAudio.audioListenerWillBeDestroyed" to
          serializer<WebAudioEvent.AudioListenerWillBeDestroyed>(),
      "WebAudio.audioNodeCreated" to serializer<WebAudioEvent.AudioNodeCreated>(),
      "WebAudio.audioNodeWillBeDestroyed" to serializer<WebAudioEvent.AudioNodeWillBeDestroyed>(),
      "WebAudio.audioParamCreated" to serializer<WebAudioEvent.AudioParamCreated>(),
      "WebAudio.audioParamWillBeDestroyed" to serializer<WebAudioEvent.AudioParamWillBeDestroyed>(),
      "WebAudio.nodesConnected" to serializer<WebAudioEvent.NodesConnected>(),
      "WebAudio.nodesDisconnected" to serializer<WebAudioEvent.NodesDisconnected>(),
      "WebAudio.nodeParamConnected" to serializer<WebAudioEvent.NodeParamConnected>(),
      "WebAudio.nodeParamDisconnected" to serializer<WebAudioEvent.NodeParamDisconnected>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<WebAudioEvent> = session.typedEvents(deserializersByEventName)

  /**
   * Notifies that a new BaseAudioContext has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextCreated)
   */
  public fun contextCreatedEvents(): Flow<WebAudioEvent.ContextCreated> =
      session.typedEvents("WebAudio.contextCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("contextCreatedEvents()"),
  )
  public fun contextCreated(): Flow<WebAudioEvent.ContextCreated> = contextCreatedEvents()

  /**
   * Notifies that an existing BaseAudioContext will be destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextWillBeDestroyed)
   */
  public fun contextWillBeDestroyedEvents(): Flow<WebAudioEvent.ContextWillBeDestroyed> =
      session.typedEvents("WebAudio.contextWillBeDestroyed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("contextWillBeDestroyedEvents()"),
  )
  public fun contextWillBeDestroyed(): Flow<WebAudioEvent.ContextWillBeDestroyed> =
      contextWillBeDestroyedEvents()

  /**
   * Notifies that existing BaseAudioContext has changed some properties (id stays the same)..
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextChanged)
   */
  public fun contextChangedEvents(): Flow<WebAudioEvent.ContextChanged> =
      session.typedEvents("WebAudio.contextChanged")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("contextChangedEvents()"),
  )
  public fun contextChanged(): Flow<WebAudioEvent.ContextChanged> = contextChangedEvents()

  /**
   * Notifies that the construction of an AudioListener has finished.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerCreated)
   */
  public fun audioListenerCreatedEvents(): Flow<WebAudioEvent.AudioListenerCreated> =
      session.typedEvents("WebAudio.audioListenerCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("audioListenerCreatedEvents()"),
  )
  public fun audioListenerCreated(): Flow<WebAudioEvent.AudioListenerCreated> =
      audioListenerCreatedEvents()

  /**
   * Notifies that a new AudioListener has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerWillBeDestroyed)
   */
  public fun audioListenerWillBeDestroyedEvents(): Flow<WebAudioEvent.AudioListenerWillBeDestroyed>
      = session.typedEvents("WebAudio.audioListenerWillBeDestroyed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("audioListenerWillBeDestroyedEvents()"),
  )
  public fun audioListenerWillBeDestroyed(): Flow<WebAudioEvent.AudioListenerWillBeDestroyed> =
      audioListenerWillBeDestroyedEvents()

  /**
   * Notifies that a new AudioNode has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeCreated)
   */
  public fun audioNodeCreatedEvents(): Flow<WebAudioEvent.AudioNodeCreated> =
      session.typedEvents("WebAudio.audioNodeCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("audioNodeCreatedEvents()"),
  )
  public fun audioNodeCreated(): Flow<WebAudioEvent.AudioNodeCreated> = audioNodeCreatedEvents()

  /**
   * Notifies that an existing AudioNode has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeWillBeDestroyed)
   */
  public fun audioNodeWillBeDestroyedEvents(): Flow<WebAudioEvent.AudioNodeWillBeDestroyed> =
      session.typedEvents("WebAudio.audioNodeWillBeDestroyed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("audioNodeWillBeDestroyedEvents()"),
  )
  public fun audioNodeWillBeDestroyed(): Flow<WebAudioEvent.AudioNodeWillBeDestroyed> =
      audioNodeWillBeDestroyedEvents()

  /**
   * Notifies that a new AudioParam has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamCreated)
   */
  public fun audioParamCreatedEvents(): Flow<WebAudioEvent.AudioParamCreated> =
      session.typedEvents("WebAudio.audioParamCreated")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("audioParamCreatedEvents()"),
  )
  public fun audioParamCreated(): Flow<WebAudioEvent.AudioParamCreated> = audioParamCreatedEvents()

  /**
   * Notifies that an existing AudioParam has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamWillBeDestroyed)
   */
  public fun audioParamWillBeDestroyedEvents(): Flow<WebAudioEvent.AudioParamWillBeDestroyed> =
      session.typedEvents("WebAudio.audioParamWillBeDestroyed")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("audioParamWillBeDestroyedEvents()"),
  )
  public fun audioParamWillBeDestroyed(): Flow<WebAudioEvent.AudioParamWillBeDestroyed> =
      audioParamWillBeDestroyedEvents()

  /**
   * Notifies that two AudioNodes are connected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesConnected)
   */
  public fun nodesConnectedEvents(): Flow<WebAudioEvent.NodesConnected> =
      session.typedEvents("WebAudio.nodesConnected")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("nodesConnectedEvents()"),
  )
  public fun nodesConnected(): Flow<WebAudioEvent.NodesConnected> = nodesConnectedEvents()

  /**
   * Notifies that AudioNodes are disconnected. The destination can be null, and it means all the
   * outgoing connections from the source are disconnected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesDisconnected)
   */
  public fun nodesDisconnectedEvents(): Flow<WebAudioEvent.NodesDisconnected> =
      session.typedEvents("WebAudio.nodesDisconnected")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("nodesDisconnectedEvents()"),
  )
  public fun nodesDisconnected(): Flow<WebAudioEvent.NodesDisconnected> = nodesDisconnectedEvents()

  /**
   * Notifies that an AudioNode is connected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamConnected)
   */
  public fun nodeParamConnectedEvents(): Flow<WebAudioEvent.NodeParamConnected> =
      session.typedEvents("WebAudio.nodeParamConnected")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("nodeParamConnectedEvents()"),
  )
  public fun nodeParamConnected(): Flow<WebAudioEvent.NodeParamConnected> =
      nodeParamConnectedEvents()

  /**
   * Notifies that an AudioNode is disconnected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamDisconnected)
   */
  public fun nodeParamDisconnectedEvents(): Flow<WebAudioEvent.NodeParamDisconnected> =
      session.typedEvents("WebAudio.nodeParamDisconnected")

  @Deprecated(
    message = "Events subscription methods were renamed with the -Events suffix.",
    replaceWith = ReplaceWith("nodeParamDisconnectedEvents()"),
  )
  public fun nodeParamDisconnected(): Flow<WebAudioEvent.NodeParamDisconnected> =
      nodeParamDisconnectedEvents()

  /**
   * Enables the WebAudio domain and starts sending context lifetime events.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-enable)
   */
  public suspend fun enable(): EnableResponse = session.request("WebAudio.enable", Unit)

  /**
   * Disables the WebAudio domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("WebAudio.disable", Unit)

  /**
   * Fetch the realtime data from the registered contexts.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-getRealtimeData)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getRealtimeData(input: GetRealtimeDataRequest): GetRealtimeDataResponse =
      session.request("WebAudio.getRealtimeData", input)

  /**
   * Fetch the realtime data from the registered contexts.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#method-getRealtimeData)
   */
  public suspend fun getRealtimeData(contextId: GraphObjectId): GetRealtimeDataResponse {
    val input = GetRealtimeDataRequest(contextId)
    return getRealtimeData(input)
  }
}
