@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.webaudio.events

import kotlin.Double
import kotlin.Suppress
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.webaudio.AudioListener
import org.hildan.chrome.devtools.domains.webaudio.AudioNode
import org.hildan.chrome.devtools.domains.webaudio.AudioParam
import org.hildan.chrome.devtools.domains.webaudio.BaseAudioContext
import org.hildan.chrome.devtools.domains.webaudio.GraphObjectId

@Serializable
public sealed class WebAudioEvent {
  /**
   * Notifies that a new BaseAudioContext has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextCreated)
   */
  @Serializable
  public data class ContextCreated(
    public val context: BaseAudioContext,
  ) : WebAudioEvent()

  /**
   * Notifies that an existing BaseAudioContext will be destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextWillBeDestroyed)
   */
  @Serializable
  public data class ContextWillBeDestroyed(
    public val contextId: GraphObjectId,
  ) : WebAudioEvent()

  /**
   * Notifies that existing BaseAudioContext has changed some properties (id stays the same)..
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextChanged)
   */
  @Serializable
  public data class ContextChanged(
    public val context: BaseAudioContext,
  ) : WebAudioEvent()

  /**
   * Notifies that the construction of an AudioListener has finished.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerCreated)
   */
  @Serializable
  public data class AudioListenerCreated(
    public val listener: AudioListener,
  ) : WebAudioEvent()

  /**
   * Notifies that a new AudioListener has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerWillBeDestroyed)
   */
  @Serializable
  public data class AudioListenerWillBeDestroyed(
    public val contextId: GraphObjectId,
    public val listenerId: GraphObjectId,
  ) : WebAudioEvent()

  /**
   * Notifies that a new AudioNode has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeCreated)
   */
  @Serializable
  public data class AudioNodeCreated(
    public val node: AudioNode,
  ) : WebAudioEvent()

  /**
   * Notifies that an existing AudioNode has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeWillBeDestroyed)
   */
  @Serializable
  public data class AudioNodeWillBeDestroyed(
    public val contextId: GraphObjectId,
    public val nodeId: GraphObjectId,
  ) : WebAudioEvent()

  /**
   * Notifies that a new AudioParam has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamCreated)
   */
  @Serializable
  public data class AudioParamCreated(
    public val `param`: AudioParam,
  ) : WebAudioEvent()

  /**
   * Notifies that an existing AudioParam has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamWillBeDestroyed)
   */
  @Serializable
  public data class AudioParamWillBeDestroyed(
    public val contextId: GraphObjectId,
    public val nodeId: GraphObjectId,
    public val paramId: GraphObjectId,
  ) : WebAudioEvent()

  /**
   * Notifies that two AudioNodes are connected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesConnected)
   */
  @Serializable
  public data class NodesConnected(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null,
    public val destinationInputIndex: Double? = null,
  ) : WebAudioEvent()

  /**
   * Notifies that AudioNodes are disconnected. The destination can be null, and it means all the
   * outgoing connections from the source are disconnected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesDisconnected)
   */
  @Serializable
  public data class NodesDisconnected(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null,
    public val destinationInputIndex: Double? = null,
  ) : WebAudioEvent()

  /**
   * Notifies that an AudioNode is connected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamConnected)
   */
  @Serializable
  public data class NodeParamConnected(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null,
  ) : WebAudioEvent()

  /**
   * Notifies that an AudioNode is disconnected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamDisconnected)
   */
  @Serializable
  public data class NodeParamDisconnected(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null,
  ) : WebAudioEvent()
}
