@file:Suppress("RedundantVisibilityModifier", "DEPRECATION", "EXPERIMENTAL_API_USAGE")

package org.hildan.chrome.devtools.domains.webauthn

import kotlin.Boolean
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.List
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi
import org.hildan.chrome.devtools.protocol.request

/**
 * Request object containing input parameters for the [WebAuthnDomain.enable] command.
 */
@Serializable
public data class EnableRequest(
  /**
   * Whether to enable the WebAuthn user interface. Enabling the UI is
   * recommended for debugging and demo purposes, as it is closer to the real
   * experience. Disabling the UI is recommended for automated testing.
   * Supported at the embedder's discretion if UI is available.
   * Defaults to false.
   */
  public val enableUI: Boolean? = null,
) {
  public class Builder() {
    /**
     * Whether to enable the WebAuthn user interface. Enabling the UI is
     * recommended for debugging and demo purposes, as it is closer to the real
     * experience. Disabling the UI is recommended for automated testing.
     * Supported at the embedder's discretion if UI is available.
     * Defaults to false.
     */
    public var enableUI: Boolean? = null

    public fun build(): EnableRequest = EnableRequest(enableUI)
  }
}

/**
 * A dummy response object for the [WebAuthnDomain.enable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object EnableResponse

/**
 * A dummy response object for the [WebAuthnDomain.disable] command. This command doesn't return any
 * result at the moment, but this could happen in the future, or could have happened in the past. For
 * forwards and backwards compatibility of the command method, we still declare this class even without
 * properties.
 */
@Serializable
public object DisableResponse

/**
 * Request object containing input parameters for the [WebAuthnDomain.addVirtualAuthenticator]
 * command.
 */
@Serializable
public data class AddVirtualAuthenticatorRequest(
  public val options: VirtualAuthenticatorOptions,
)

/**
 * Response type for the [WebAuthnDomain.addVirtualAuthenticator] command.
 */
@Serializable
public data class AddVirtualAuthenticatorResponse(
  public val authenticatorId: AuthenticatorId,
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.setResponseOverrideBits]
 * command.
 */
@Serializable
public data class SetResponseOverrideBitsRequest(
  public val authenticatorId: AuthenticatorId,
  /**
   * If isBogusSignature is set, overrides the signature in the authenticator response to be zero.
   * Defaults to false.
   */
  public val isBogusSignature: Boolean? = null,
  /**
   * If isBadUV is set, overrides the UV bit in the flags in the authenticator response to
   * be zero. Defaults to false.
   */
  public val isBadUV: Boolean? = null,
  /**
   * If isBadUP is set, overrides the UP bit in the flags in the authenticator response to
   * be zero. Defaults to false.
   */
  public val isBadUP: Boolean? = null,
) {
  public class Builder(
    public val authenticatorId: AuthenticatorId,
  ) {
    /**
     * If isBogusSignature is set, overrides the signature in the authenticator response to be zero.
     * Defaults to false.
     */
    public var isBogusSignature: Boolean? = null

    /**
     * If isBadUV is set, overrides the UV bit in the flags in the authenticator response to
     * be zero. Defaults to false.
     */
    public var isBadUV: Boolean? = null

    /**
     * If isBadUP is set, overrides the UP bit in the flags in the authenticator response to
     * be zero. Defaults to false.
     */
    public var isBadUP: Boolean? = null

    public fun build(): SetResponseOverrideBitsRequest =
        SetResponseOverrideBitsRequest(authenticatorId, isBogusSignature, isBadUV, isBadUP)
  }
}

/**
 * A dummy response object for the [WebAuthnDomain.setResponseOverrideBits] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object SetResponseOverrideBitsResponse

/**
 * Request object containing input parameters for the [WebAuthnDomain.removeVirtualAuthenticator]
 * command.
 */
@Serializable
public data class RemoveVirtualAuthenticatorRequest(
  public val authenticatorId: AuthenticatorId,
)

/**
 * A dummy response object for the [WebAuthnDomain.removeVirtualAuthenticator] command. This command
 * doesn't return any result at the moment, but this could happen in the future, or could have happened
 * in the past. For forwards and backwards compatibility of the command method, we still declare this
 * class even without properties.
 */
@Serializable
public object RemoveVirtualAuthenticatorResponse

/**
 * Request object containing input parameters for the [WebAuthnDomain.addCredential] command.
 */
@Serializable
public data class AddCredentialRequest(
  public val authenticatorId: AuthenticatorId,
  public val credential: Credential,
)

/**
 * A dummy response object for the [WebAuthnDomain.addCredential] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object AddCredentialResponse

/**
 * Request object containing input parameters for the [WebAuthnDomain.getCredential] command.
 */
@Serializable
public data class GetCredentialRequest(
  public val authenticatorId: AuthenticatorId,
  public val credentialId: String,
)

/**
 * Response type for the [WebAuthnDomain.getCredential] command.
 */
@Serializable
public data class GetCredentialResponse(
  public val credential: Credential,
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.getCredentials] command.
 */
@Serializable
public data class GetCredentialsRequest(
  public val authenticatorId: AuthenticatorId,
)

/**
 * Response type for the [WebAuthnDomain.getCredentials] command.
 */
@Serializable
public data class GetCredentialsResponse(
  public val credentials: List<Credential>,
)

/**
 * Request object containing input parameters for the [WebAuthnDomain.removeCredential] command.
 */
@Serializable
public data class RemoveCredentialRequest(
  public val authenticatorId: AuthenticatorId,
  public val credentialId: String,
)

/**
 * A dummy response object for the [WebAuthnDomain.removeCredential] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object RemoveCredentialResponse

/**
 * Request object containing input parameters for the [WebAuthnDomain.clearCredentials] command.
 */
@Serializable
public data class ClearCredentialsRequest(
  public val authenticatorId: AuthenticatorId,
)

/**
 * A dummy response object for the [WebAuthnDomain.clearCredentials] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object ClearCredentialsResponse

/**
 * Request object containing input parameters for the [WebAuthnDomain.setUserVerified] command.
 */
@Serializable
public data class SetUserVerifiedRequest(
  public val authenticatorId: AuthenticatorId,
  public val isUserVerified: Boolean,
)

/**
 * A dummy response object for the [WebAuthnDomain.setUserVerified] command. This command doesn't
 * return any result at the moment, but this could happen in the future, or could have happened in the
 * past. For forwards and backwards compatibility of the command method, we still declare this class
 * even without properties.
 */
@Serializable
public object SetUserVerifiedResponse

/**
 * Request object containing input parameters for the
 * [WebAuthnDomain.setAutomaticPresenceSimulation] command.
 */
@Serializable
public data class SetAutomaticPresenceSimulationRequest(
  public val authenticatorId: AuthenticatorId,
  public val enabled: Boolean,
)

/**
 * A dummy response object for the [WebAuthnDomain.setAutomaticPresenceSimulation] command. This
 * command doesn't return any result at the moment, but this could happen in the future, or could have
 * happened in the past. For forwards and backwards compatibility of the command method, we still
 * declare this class even without properties.
 */
@Serializable
public object SetAutomaticPresenceSimulationResponse

/**
 * This domain allows configuring virtual authenticators to test the WebAuthn
 * API.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn)
 */
@ExperimentalChromeApi
public class WebAuthnDomain internal constructor(
  private val session: ChromeDPSession,
) {
  /**
   * Enable the WebAuthn domain and start intercepting credential storage and
   * retrieval with a virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-enable)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun enable(input: EnableRequest): EnableResponse =
      session.request("WebAuthn.enable", input)

  /**
   * Enable the WebAuthn domain and start intercepting credential storage and
   * retrieval with a virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-enable)
   */
  @JvmOverloads
  public suspend inline fun enable(optionalArgs: EnableRequest.Builder.() -> Unit = {}):
      EnableResponse {
    val builder = EnableRequest.Builder()
    val input = builder.apply(optionalArgs).build()
    return enable(input)
  }

  /**
   * Disable the WebAuthn domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-disable)
   */
  public suspend fun disable(): DisableResponse = session.request("WebAuthn.disable", Unit)

  /**
   * Creates and adds a virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-addVirtualAuthenticator)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun addVirtualAuthenticator(input: AddVirtualAuthenticatorRequest):
      AddVirtualAuthenticatorResponse = session.request("WebAuthn.addVirtualAuthenticator", input)

  /**
   * Creates and adds a virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-addVirtualAuthenticator)
   */
  public suspend fun addVirtualAuthenticator(options: VirtualAuthenticatorOptions):
      AddVirtualAuthenticatorResponse {
    val input = AddVirtualAuthenticatorRequest(options)
    return addVirtualAuthenticator(input)
  }

  /**
   * Resets parameters isBogusSignature, isBadUV, isBadUP to false if they are not present.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setResponseOverrideBits)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setResponseOverrideBits(input: SetResponseOverrideBitsRequest):
      SetResponseOverrideBitsResponse = session.request("WebAuthn.setResponseOverrideBits", input)

  /**
   * Resets parameters isBogusSignature, isBadUV, isBadUP to false if they are not present.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setResponseOverrideBits)
   */
  @JvmOverloads
  public suspend inline fun setResponseOverrideBits(authenticatorId: AuthenticatorId,
      optionalArgs: SetResponseOverrideBitsRequest.Builder.() -> Unit = {}):
      SetResponseOverrideBitsResponse {
    val builder = SetResponseOverrideBitsRequest.Builder(authenticatorId)
    val input = builder.apply(optionalArgs).build()
    return setResponseOverrideBits(input)
  }

  /**
   * Removes the given authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-removeVirtualAuthenticator)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeVirtualAuthenticator(input: RemoveVirtualAuthenticatorRequest):
      RemoveVirtualAuthenticatorResponse = session.request("WebAuthn.removeVirtualAuthenticator",
      input)

  /**
   * Removes the given authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-removeVirtualAuthenticator)
   */
  public suspend fun removeVirtualAuthenticator(authenticatorId: AuthenticatorId):
      RemoveVirtualAuthenticatorResponse {
    val input = RemoveVirtualAuthenticatorRequest(authenticatorId)
    return removeVirtualAuthenticator(input)
  }

  /**
   * Adds the credential to the specified authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-addCredential)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun addCredential(input: AddCredentialRequest): AddCredentialResponse =
      session.request("WebAuthn.addCredential", input)

  /**
   * Adds the credential to the specified authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-addCredential)
   */
  public suspend fun addCredential(authenticatorId: AuthenticatorId, credential: Credential):
      AddCredentialResponse {
    val input = AddCredentialRequest(authenticatorId, credential)
    return addCredential(input)
  }

  /**
   * Returns a single credential stored in the given virtual authenticator that
   * matches the credential ID.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-getCredential)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getCredential(input: GetCredentialRequest): GetCredentialResponse =
      session.request("WebAuthn.getCredential", input)

  /**
   * Returns a single credential stored in the given virtual authenticator that
   * matches the credential ID.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-getCredential)
   */
  public suspend fun getCredential(authenticatorId: AuthenticatorId, credentialId: String):
      GetCredentialResponse {
    val input = GetCredentialRequest(authenticatorId, credentialId)
    return getCredential(input)
  }

  /**
   * Returns all the credentials stored in the given virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-getCredentials)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun getCredentials(input: GetCredentialsRequest): GetCredentialsResponse =
      session.request("WebAuthn.getCredentials", input)

  /**
   * Returns all the credentials stored in the given virtual authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-getCredentials)
   */
  public suspend fun getCredentials(authenticatorId: AuthenticatorId): GetCredentialsResponse {
    val input = GetCredentialsRequest(authenticatorId)
    return getCredentials(input)
  }

  /**
   * Removes a credential from the authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-removeCredential)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun removeCredential(input: RemoveCredentialRequest): RemoveCredentialResponse =
      session.request("WebAuthn.removeCredential", input)

  /**
   * Removes a credential from the authenticator.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-removeCredential)
   */
  public suspend fun removeCredential(authenticatorId: AuthenticatorId, credentialId: String):
      RemoveCredentialResponse {
    val input = RemoveCredentialRequest(authenticatorId, credentialId)
    return removeCredential(input)
  }

  /**
   * Clears all the credentials from the specified device.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-clearCredentials)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun clearCredentials(input: ClearCredentialsRequest): ClearCredentialsResponse =
      session.request("WebAuthn.clearCredentials", input)

  /**
   * Clears all the credentials from the specified device.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-clearCredentials)
   */
  public suspend fun clearCredentials(authenticatorId: AuthenticatorId): ClearCredentialsResponse {
    val input = ClearCredentialsRequest(authenticatorId)
    return clearCredentials(input)
  }

  /**
   * Sets whether User Verification succeeds or fails for an authenticator.
   * The default is true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setUserVerified)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setUserVerified(input: SetUserVerifiedRequest): SetUserVerifiedResponse =
      session.request("WebAuthn.setUserVerified", input)

  /**
   * Sets whether User Verification succeeds or fails for an authenticator.
   * The default is true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setUserVerified)
   */
  public suspend fun setUserVerified(authenticatorId: AuthenticatorId, isUserVerified: Boolean):
      SetUserVerifiedResponse {
    val input = SetUserVerifiedRequest(authenticatorId, isUserVerified)
    return setUserVerified(input)
  }

  /**
   * Sets whether tests of user presence will succeed immediately (if true) or fail to resolve (if
   * false) for an authenticator.
   * The default is true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setAutomaticPresenceSimulation)
   *
   * Note: this function uses an input class, and constructing this class manually may lead to
   * incompatibilities if the class's constructor arguments change in the future. For maximum
   * compatibility, it is advised to use the overload of this function that directly takes the
   * mandatory parameters as arguments, and the optional ones from a configuration lambda.
   */
  public suspend fun setAutomaticPresenceSimulation(input: SetAutomaticPresenceSimulationRequest):
      SetAutomaticPresenceSimulationResponse =
      session.request("WebAuthn.setAutomaticPresenceSimulation", input)

  /**
   * Sets whether tests of user presence will succeed immediately (if true) or fail to resolve (if
   * false) for an authenticator.
   * The default is true.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAuthn/#method-setAutomaticPresenceSimulation)
   */
  public suspend fun setAutomaticPresenceSimulation(authenticatorId: AuthenticatorId,
      enabled: Boolean): SetAutomaticPresenceSimulationResponse {
    val input = SetAutomaticPresenceSimulationRequest(authenticatorId, enabled)
    return setAutomaticPresenceSimulation(input)
  }
}
