/*
 * Decompiled with CFR 0.152.
 */
package org.hildan.hashcode.utils.runner;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.hildan.hashcode.utils.runner.ExceptionLoggingExecutorService;
import org.hildan.hashcode.utils.runner.UncaughtExceptionsPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HCRunner<I> {
    private static final Logger logger = LoggerFactory.getLogger(HCRunner.class);
    private final Consumer<I> solver;
    private final List<Throwable> exceptions;
    private final UncaughtExceptionsPolicy exceptionsPolicy;

    public HCRunner(Consumer<I> solver, UncaughtExceptionsPolicy exceptionsPolicy) {
        this.solver = solver;
        this.exceptionsPolicy = exceptionsPolicy;
        this.exceptions = new ArrayList<Throwable>();
    }

    @SafeVarargs
    public final void run(I ... inputs) {
        this.run(inputs.length, inputs);
    }

    @SafeVarargs
    public final void run(int nThreads, I ... inputs) {
        if (inputs.length < 1) {
            throw new IllegalArgumentException("No input passed as argument");
        }
        ExceptionLoggingExecutorService threadPool = new ExceptionLoggingExecutorService(nThreads, this.exceptionsPolicy);
        List<Future<?>> futures = this.submitInputs(inputs, threadPool);
        this.waitForTermination(futures);
        this.shutdownAndWaitForTermination(threadPool);
        this.remindExceptions(inputs);
    }

    private List<Future<?>> submitInputs(I[] inputs, ExecutorService threadPool) {
        ArrayList handles = new ArrayList(inputs.length);
        for (I input : inputs) {
            Future<?> future = threadPool.submit(() -> this.solver.accept(input));
            handles.add(future);
        }
        return handles;
    }

    private void waitForTermination(List<Future<?>> tasks) {
        for (Future<?> task : tasks) {
            try {
                task.get();
            }
            catch (ExecutionException e) {
                this.exceptions.add(e.getCause());
            }
            catch (InterruptedException e) {
                logger.error("Interrupted while waiting for tasks to complete", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        }
    }

    private void shutdownAndWaitForTermination(ExecutorService threadPool) {
        try {
            threadPool.shutdown();
            threadPool.awaitTermination(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            logger.error("Interrupted while waiting for thread pool to shut down", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private void remindExceptions(I[] inputs) {
        if (this.exceptions.isEmpty()) {
            return;
        }
        if (this.exceptionsPolicy.shouldLogViaSlf4J()) {
            this.logExceptions(inputs);
        }
        if (this.exceptionsPolicy.shouldPrintOnStdErr()) {
            this.printExceptionsOnStdErr(inputs);
        }
    }

    private void logExceptions(I[] inputs) {
        logger.error("{} tasks terminated abruptly by throwing exceptions", (Object)this.exceptions.size());
        for (int i = 0; i < this.exceptions.size(); ++i) {
            Throwable e = this.exceptions.get(i);
            logger.error("Reminder: this exception was thrown while running on input " + inputs[i] + ":", e);
        }
    }

    private void printExceptionsOnStdErr(I[] inputs) {
        System.err.println(this.exceptions.size() + " tasks terminated abruptly by throwing exceptions");
        for (int i = 0; i < this.exceptions.size(); ++i) {
            Throwable e = this.exceptions.get(i);
            System.err.println("Reminder: this exception was thrown while running on input " + inputs[i] + ":");
            e.printStackTrace();
        }
    }
}

