/*
 * Copyright(c) 2016 - 2020, Clouds Studio Holding Limited. All rights reserved.
 * Project : components
 * File : DistributedSequence.java
 * Date : 7/22/20, 12:51 AM
 * Author : Hsi Chu
 * Contact : hiylo@live.com
 */

package org.hiylo.components.sequence;


import org.hiylo.components.zookeeper.I0ItecZooKeeperServiceProvider;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.List;

/**
 * 基于Twitter的Snowflake算法实现分布式高效有序ID生产黑科技(sequence)
 *
 * <br>
 * SnowFlake的结构如下(每部分用-分开):<br>
 * <br>
 * 0 - 0000000000 0000000000 0000000000 0000000000 0 - 00000 - 00000 - 000000000000 <br>
 * <br>
 * 1位标识，由于long基本类型在Java中是带符号的，最高位是符号位，正数是0，负数是1，所以id一般是正数，最高位是0<br>
 * <br>
 * 41位时间截(毫秒级)，注意，41位时间截不是存储当前时间的时间截，而是存储时间截的差值（当前时间截 - 开始时间截)
 * 得到的值），这里的的开始时间截，一般是我们的id生成器开始使用的时间，由我们程序来指定的（如下下面程序IdWorker类的startTime属性）。41位的时间截，可以使用69年，年T = (1L << 41) / (1000L * 60 * 60 * 24 * 365) = 69<br>
 * <br>
 * 10位的数据机器位，可以部署在1024个节点，包括5位datacenterId和5位workerId<br>
 * <br>
 * 12位序列，毫秒内的计数，12位的计数顺序号支持每个节点每毫秒(同一机器，同一时间截)产生4096个ID序号<br>
 * <br>
 * <br>
 * 加起来刚好64位，为一个Long型。<br>
 * SnowFlake的优点是，整体上按照时间自增排序，并且整个分布式系统内不会产生ID碰撞(由数据中心ID和机器ID作区分)，并且效率较高，经测试，SnowFlake每秒能够产生26万ID左右。
 *
 * @author hiylo
 * @date 2018年2月13日 00:22:51
 */
public class DistributedSequence extends AbstractSequence {

    @Autowired
    private I0ItecZooKeeperServiceProvider zooKeeperServiceProvider;
    /**
     * 工作机器ID(0~31)
     */
    private int workerId;
    /**
     * 数据中心ID(0~31)
     */
    private int datacenterId;
    /**
     * 毫秒内序列(0~4095)
     */
    private long sequence = 0L;
    private SingletonSequence singletonSequence;

    public DistributedSequence() {
        this.workerId = 0;
        this.datacenterId = 0;
    }

    @PostConstruct
    public void init() {
        try {
            List<String> datacenterIds = zooKeeperServiceProvider.get("/config/sequence/");
            for (int i = 0; i <= maxWorkerId; i++) {
                if (datacenterIds.contains(String.valueOf(i))) {
                    List<String> workIds = zooKeeperServiceProvider.get("/config/sequence/" + i + "/worker/");
                    for (int j = 0; j <= maxWorkerId; j++) {
                        if (!workIds.contains(String.valueOf(i))) {
                            this.workerId = i;
                            this.datacenterId = i;
                            zooKeeperServiceProvider.put("/config/sequence/" + i + "/worker/", String.valueOf(i));
                            singletonSequence = new SingletonSequence(datacenterId, workerId);
                            return;
                        }
                    }
                } else {
                    this.workerId = i;
                    this.datacenterId = 0;
                    zooKeeperServiceProvider.put("/config/sequence/" + i + "/worker/", String.valueOf(0));
                    singletonSequence = new SingletonSequence(datacenterId, workerId);
                    return;
                }
            }
        } catch (RuntimeException e) {
            zooKeeperServiceProvider.put("/config/sequence/0/worker/", "0");
            this.workerId = 0;
            this.datacenterId = 0;
            singletonSequence = new SingletonSequence(datacenterId, workerId);
            return;
        }
        throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
    }

    /**
     * 获得下一个ID (该方法是线程安全的)
     *
     * @return
     */
    @Override
    public synchronized String nextIdString() {
        return singletonSequence.nextIdString();
    }

    /**
     * 获得下一个ID (该方法是线程安全的)
     *
     * @return
     */
    @Override
    public synchronized long nextId() {
        return singletonSequence.nextId();
    }

}