package org.honton.chas.manifest.dependencies;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CoderResult;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.versioning.OverConstrainedVersionException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;

/**
 * Create a manifest file with entries for every project dependency.  The name attribute of each
 * section is the dependency Group:Artifact[:Classifier]:Packaging.  The second attribute is the
 * scope and version of that dependency. e.g.:
 * <pre>
 Name: junit:junit:jar
 test: 3.8.1
 * </pre>
 * The output manifest can be used as the manifestFile parameter to the maven archiver.
 */
@Mojo(name = "manifest", requiresDependencyResolution = ResolutionScope.TEST, defaultPhase = LifecyclePhase.INITIALIZE)
public class DependencyManifest extends AbstractMojo {

  final static byte[] CRLF = "\r\n".getBytes(StandardCharsets.UTF_8);
  final static byte[] SPACE = " ".getBytes(StandardCharsets.UTF_8);

  /**
   * The current Maven project
   */
  @Parameter(defaultValue = "${project}", readonly = true, required = true)
  private MavenProject mavenProject;

  /**
   * Skip dependency manifest plugin execution
   */
  @Parameter(property = "manifest.skip", defaultValue = "false")
  private boolean skip;

  /**
   * Location of output manifest file
   */
  @Parameter(property = "manifest.output", defaultValue = "${project.build.directory}/classpath.manifest")
  private File manifest;

  private CharsetEncoder encoder = StandardCharsets.UTF_8.newEncoder();
  private ByteBuffer out = ByteBuffer.allocate(72);

  /**
   * Main entry into mojo. Gets the list of dependencies and iterates through setting a property for
   * each artifact.
   *
   * @throws MojoFailureException with a message if an error occurs.
   */
  @Override public void execute() throws MojoFailureException {
    if (skip) {
      getLog().info("Skipping plugin execution");
      return;
    }

    Set<Artifact> artifacts = mavenProject.getArtifacts();
    if (artifacts.isEmpty()) {
      getLog().info("No dependencies");
      return;
    }

    try {
      writeManifest(buildManifest());
    } catch (OverConstrainedVersionException | IOException e) {
      throw new MojoFailureException(e.getMessage(), e);
    }
  }

  private Map<String,Artifact> buildManifest() throws OverConstrainedVersionException {
    Map<String,Artifact> manifest = new TreeMap<>();
    Set<Artifact> artifacts = mavenProject.getArtifacts();
    for (Artifact artifact : artifacts) {
      manifest.put(artifact.getDependencyConflictId().toString(), artifact);
    }
    return manifest;
  }

  private void writeManifest(Map<String,Artifact> artifactMap) throws IOException, OverConstrainedVersionException {
    File dir = manifest.getParentFile();
    if (!dir.exists() && !dir.mkdirs()) {
      throw new IOException("Unable to create " + dir);
    }
    FileOutputStream fos = new FileOutputStream(manifest);
    FileChannel fileChannel = fos.getChannel();

    for(Entry<String, Artifact> entry : artifactMap.entrySet()) {
      writePair(fileChannel, "");
      writePair(fileChannel, "Name: " + entry.getKey());
      Artifact artifact = entry.getValue();
      writePair(fileChannel, artifact.getScope() + ": " + artifact.getSelectedVersion());
    }
    fileChannel.close();
  }

  /**
   * Adds line breaks to enforce a maximum 72 bytes per line.
   */
  void writePair(FileChannel fileChannel, String name_value) throws IOException {
    CharBuffer in = CharBuffer.wrap(name_value);
    out.limit(70);
    CoderResult result = flush(fileChannel, in);
    while(result.isOverflow()) {
      out.limit(70);
      out.put(SPACE);
      result = flush(fileChannel, in);
    }
  }

  private CoderResult flush(FileChannel fileChannel, CharBuffer in) throws IOException {
    CoderResult result = encoder.encode(in, out, true);
    out.limit(72);
    out.put(CRLF);
    out.flip();
    fileChannel.write(out);
    out.rewind();
    return result;
  }
}
