/*
 * Copyright 2009 Red Hat, Inc.
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *    http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */

package org.hornetq.jms.client;

import java.io.Serializable;
import java.util.UUID;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.TemporaryQueue;
import javax.jms.TemporaryTopic;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.Referenceable;

import org.hornetq.api.core.Pair;
import org.hornetq.api.core.SimpleString;
import org.hornetq.jms.referenceable.DestinationObjectFactory;
import org.hornetq.jms.referenceable.SerializableObjectRefAddr;

/**
 * HornetQ implementation of a JMS Destination.
 * 
 * @author <a href="mailto:tim.fox@jboss.com">Tim Fox</a>
 * @version <tt>$Revision: 8859 $</tt>
 *
 * $Id: HornetQDestination.java 8859 2010-01-29 18:51:42Z timfox $
 */
public class HornetQDestination implements TemporaryQueue, TemporaryTopic, Serializable, Referenceable
{
   // Constants -----------------------------------------------------

   // Static --------------------------------------------------------

   /**
    * 
    */
   private static final long serialVersionUID = 5027962425462382883L;
   
   public static final String JMS_QUEUE_ADDRESS_PREFIX = "jms.queue.";
   
   public static final String JMS_TOPIC_ADDRESS_PREFIX = "jms.topic.";
   
   private static final char SEPARATOR = '.';
      
   protected static String escape(final String input)
   {
      if (input == null)
      {
         return "";
      }
      return input.replace("\\", "\\\\").replace(".", "\\.");
   }

   public static Destination fromAddress(final String address)
   {
      if (address.startsWith(HornetQDestination.JMS_QUEUE_ADDRESS_PREFIX))
      {
         String name = address.substring(HornetQDestination.JMS_QUEUE_ADDRESS_PREFIX.length());

         return createQueue(name);
      }
      else if (address.startsWith(HornetQDestination.JMS_TOPIC_ADDRESS_PREFIX))
      {
         String name = address.substring(HornetQDestination.JMS_TOPIC_ADDRESS_PREFIX.length());

         return createTopic(name);
      }
      else
      {
         throw new IllegalArgumentException("Invalid address " + address);
      }
   }
   
   public static String createQueueNameForDurableSubscription(final String clientID, final String subscriptionName)
   {
      return HornetQDestination.escape(clientID) + SEPARATOR + HornetQDestination.escape(subscriptionName);
   }
   
   public static Pair<String, String> decomposeQueueNameForDurableSubscription(final String queueName)
   {
      StringBuffer[] parts = new StringBuffer[2];
      int currentPart = 0;

      parts[0] = new StringBuffer();
      parts[1] = new StringBuffer();

      int pos = 0;
      while (pos < queueName.length())
      {
         char ch = queueName.charAt(pos);
         pos++;

         if (ch == SEPARATOR)
         {
            currentPart++;
            if (currentPart >= parts.length)
            {
               throw new IllegalArgumentException("Invalid message queue name: " + queueName);
            }

            continue;
         }

         if (ch == '\\')
         {
            if (pos >= queueName.length())
            {
               throw new IllegalArgumentException("Invalid message queue name: " + queueName);
            }
            ch = queueName.charAt(pos);
            pos++;
         }

         parts[currentPart].append(ch);
      }

      if (currentPart != 1)
      {
         throw new IllegalArgumentException("Invalid message queue name: " + queueName);
      }

      Pair<String, String> pair = new Pair<String, String>(parts[0].toString(), parts[1].toString());

      return pair;
   }
   
   public static SimpleString createQueueAddressFromName(final String name)
   {
      return new SimpleString(JMS_QUEUE_ADDRESS_PREFIX + name);
   }
   
   public static SimpleString createTopicAddressFromName(final String name)
   {
      return new SimpleString(JMS_TOPIC_ADDRESS_PREFIX + name);
   }
   
   public static HornetQDestination createQueue(final String name)
   {
      return new HornetQDestination(JMS_QUEUE_ADDRESS_PREFIX.concat(name), name, false, true, null);
   }
   
   public static HornetQDestination createTopic(final String name)
   {
      return new HornetQDestination(JMS_TOPIC_ADDRESS_PREFIX.concat(name), name, false, false, null);
   }
   
   public static HornetQDestination createTemporaryQueue(final HornetQSession session)
   {
      String name = UUID.randomUUID().toString();
      
      return new HornetQDestination(JMS_QUEUE_ADDRESS_PREFIX.concat(name), name, true, true, session);
   }
   
   public static HornetQDestination createTemporaryTopic(final HornetQSession session)
   {
      String name = UUID.randomUUID().toString();
      
      return new HornetQDestination(JMS_TOPIC_ADDRESS_PREFIX.concat(name), name, true, false, session);
   }

   
   // Attributes ----------------------------------------------------

   /**
    * The JMS name
    */
   protected final String name;

   /**
    * The core address
    */
   private final String address;

   /**
    * SimpleString version of address
    */
   private final SimpleString simpleAddress;
   
   private final boolean temporary;
   
   private final boolean queue;
   
   private final HornetQSession session;
   
   // Constructors --------------------------------------------------

   private HornetQDestination(final String address, final String name,
                                final boolean temporary,
                                final boolean queue,
                                final HornetQSession session)
   {
      this.address = address;

      this.name = name;

      simpleAddress = new SimpleString(address);
      
      this.temporary = temporary;
      
      this.queue = queue;
      
      this.session = session;
   }
   
   // Referenceable implementation ---------------------------------------

   public Reference getReference() throws NamingException
   {
      return new Reference(this.getClass().getCanonicalName(),
                           new SerializableObjectRefAddr("HornetQ-DEST", this),
                           DestinationObjectFactory.class.getCanonicalName(),
                           null);
   }

   public String getQueueName()
   {
      return name;
   }
   
   public String getTopicName()
   {
      return name;
   }
   
   public void delete() throws JMSException
   {
      if (session != null)
      {
         if (queue)
         {
            session.deleteTemporaryQueue(this);
         }
         else
         {
            session.deleteTemporaryTopic(this);
         }
      }
   }
   
   public boolean isQueue()
   {
      return queue;
   }
   
   // Public --------------------------------------------------------

   public String getAddress()
   {
      return address;
   }

   public SimpleString getSimpleAddress()
   {
      return simpleAddress;
   }

   public String getName()
   {
      return name;
   }

   public boolean isTemporary()
   {
      return temporary;
   }
      
   @Override
   public boolean equals(final Object o)
   {
      if (this == o)
      {
         return true;
      }

      if (!(o instanceof HornetQDestination))
      {
         return false;
      }

      HornetQDestination that = (HornetQDestination)o;

      return address.equals(that.address);
   }

   @Override
   public int hashCode()
   {
      return address.hashCode();
   }

   // Package protected ---------------------------------------------

   // Protected -----------------------------------------------------

   // Private -------------------------------------------------------

   // Inner classes -------------------------------------------------
}
