package org.hotrod.dynamicsql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class DynamicCursor<R> implements Cursor<R> {

  private static final Logger log = Logger.getLogger(DynamicCursor.class.getName());

  private Connection conn;
  private PreparedStatement ps;
  private ResultSet rs;
  private RowReader<R> rowReader;

  public DynamicCursor(Connection conn, PreparedSelectQuery<R> q, RowReader<R> rowReader, final Integer fetchSize)
      throws SQLException {

    try {

      this.ps = q.prepareStatement(conn);
      if (fetchSize != null) {
        this.ps.setFetchSize(fetchSize);
      }

      q.applyParameters(this.ps, conn);

      this.rs = this.ps.executeQuery();

      if (rowReader == null) {
        this.rowReader = new DynRowReader(rs.getMetaData());
      } else {
        this.rowReader = rowReader;
      }
      this.rowReader.discoverColumns(this.rs);

    } catch (SQLException e) {
      this.close();
      throw e;
    }

  }

  @Override
  public Iterator<R> iterator() {
    return new CursorIterator<>(this.conn, this.rs, this.rowReader);
  }

  @Override
  public void close() {
    try {
      try {
        if (this.rs != null) {
          this.rs.close();
        }
      } catch (SQLException e) {
        log.log(Level.SEVERE, "Could not close the database result set", e);
      }
    } finally {
      try {
        if (this.ps != null) {
          this.ps.close();
        }
      } catch (SQLException e) {
        log.log(Level.SEVERE, "Could not close the database prepared statement", e);
      } finally {
        try {
          if (this.conn != null) {
            this.conn.close();
          }
        } catch (SQLException e) {
          log.log(Level.SEVERE, "Could not close the database connection", e);
        }
      }
    }
  }

  public static class CursorIterator<T> implements Iterator<T> {

    private Connection conn;
    private ResultSet rs;
    private RowReader<T> rowReader;
    private boolean endReached;

    public CursorIterator(Connection conn, ResultSet rs, RowReader<T> rowReader) {
      this.conn = conn;
      this.rs = rs;
      this.rowReader = rowReader;
      this.endReached = false;
    }

    @Override
    public boolean hasNext() {
      if (this.endReached) {
        throw new NoSuchElementException("There are no more rows in the result set");
      }
      try {
        boolean next = this.rs.next();
        if (!next) {
          this.endReached = true;
        }
        return next;
      } catch (SQLException e) {
        throw new RuntimeException("Could not advance to the next row of the result set", e);
      }
    }

    @Override
    public T next() {
      if (this.endReached) {
        throw new NoSuchElementException("There are no more rows in the result set");
      }
      try {
        return this.rowReader.readRowFrom(this.rs, this.conn);
      } catch (SQLException e) {
        throw new RuntimeException("Could not read the result set row", e);
      }
    }

  }

  class DynRowReader implements RowReader<R> {

    private List<String> columns = null;

    public DynRowReader(ResultSetMetaData rm) throws SQLException {
      int columnCount = rm.getColumnCount();
      this.columns = new ArrayList<>();
      for (int i = 1; i <= columnCount; i++) {
        columns.add(rm.getColumnName(i));
      }
    }

    @Override
    public void discoverColumns(ResultSet rs) throws SQLException {
      // nothing to do; all columns designated
    }

    @SuppressWarnings("unchecked")
    @Override
    public R readRowFrom(ResultSet rs, Connection conn) throws SQLException {
      Row row = new Row();
      int i = 1;
      for (String column : this.columns) {
        Object value = rs.getObject(i);
        row.put(column, value);
        i++;
      }
      return (R) row;
    }

  }

}
