/*
 * Decompiled with CFR 0.152.
 */
package org.hotrod.torcs.ctp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.OptionalInt;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;

public class LogResistantFormatter {
    private static final int DEFAULT_SEGMENT_SIZE = 120;
    private static final int MIN_SEGMENT_SIZE = 40;
    private static final int MAX_SEGMENT_SIZE = 100000;
    private static final String MESSAGE_DIGEST_ALGORITHM = "SHA-256";
    private static final boolean USE_FAST_GZIP = true;
    private static final int COMPRESSION_LEVEL = 5;
    private int segmentSize;

    public LogResistantFormatter() {
        this.segmentSize = 120;
    }

    public LogResistantFormatter(int segmentSize) {
        this.setSegmentSize(segmentSize);
    }

    public void setSegmentSize(int segmentSize) {
        this.segmentSize = segmentSize;
        if (segmentSize < 40) {
            throw new IllegalArgumentException("Segment size must be greater or equal to 40, but it's " + segmentSize + ".");
        }
        if (segmentSize > 100000) {
            throw new IllegalArgumentException("Segment size must less or equal to 100000, but it's " + segmentSize + ".");
        }
    }

    public List<String> render(String plan) throws IOException {
        byte[] binaryHashed = this.encodeAndHash(plan);
        byte[] compressed = LogResistantFormatter.compress(binaryHashed, 5, true);
        String base64 = this.encodeBase64(compressed);
        List<String> lines = this.splitAndShield(base64);
        return lines;
    }

    public String parse(String[] logLines) throws InvalidLogDataException {
        byte[] hashed;
        List<Segment> segments = this.extractUnshield(logLines);
        if (segments.isEmpty()) {
            throw new InvalidLogDataException("There are no log segments in this log.");
        }
        List<Segment> deduplicated = this.deduplicate(segments);
        String consolidated = this.validateSortConsolidate(deduplicated);
        byte[] compressed = this.decodeBase64(consolidated);
        try {
            hashed = LogResistantFormatter.decompress(compressed, true);
        }
        catch (IOException | DataFormatException e) {
            throw new InvalidLogDataException("Could not decompress data: " + e.getMessage());
        }
        String plan = this.validateHashRestore(hashed);
        return plan;
    }

    private List<Segment> extractUnshield(String[] logLines) {
        String log = Arrays.stream(logLines).collect(Collectors.joining());
        Pattern p = Pattern.compile("\\{([0-9]+)/([0-9]+)\\:([^\\}]*)}*");
        ArrayList<Segment> segments = new ArrayList<Segment>();
        Matcher m = p.matcher(log);
        while (m.find()) {
            String line = m.group(1);
            String total = m.group(2);
            String content = m.group(3);
            segments.add(new Segment(Integer.valueOf(line), Integer.valueOf(total), content));
        }
        return segments;
    }

    private List<Segment> deduplicate(List<Segment> segments) throws InvalidLogDataException {
        HashMap<Integer, Segment> byLine = new HashMap<Integer, Segment>();
        for (Segment s : segments) {
            Segment existing = (Segment)byLine.get(s.getLine());
            if (existing == null) {
                byLine.put(s.getLine(), s);
                continue;
            }
            if (existing.getTotal() != s.getTotal()) {
                throw new InvalidLogDataException("Multiple segments found for line #" + s.getLine() + ", but with different total values (" + s.getTotal() + " and " + existing.getTotal() + ").");
            }
            if (this.equalStrings(existing.getContent(), s.getContent())) continue;
            throw new InvalidLogDataException("Multiple segments found for line #" + s.getLine() + ", but with different content.");
        }
        return new ArrayList<Segment>(byLine.values());
    }

    private boolean equalStrings(String a, String b) {
        return a == null ? b == null : a.equals(b);
    }

    private String validateSortConsolidate(List<Segment> segments) throws InvalidLogDataException {
        OptionalInt minTotal = segments.stream().mapToInt(s -> s.getTotal()).min();
        OptionalInt maxTotal = segments.stream().mapToInt(s -> s.getTotal()).max();
        if (minTotal.getAsInt() != maxTotal.getAsInt()) {
            throw new InvalidLogDataException("All segments should have the same total value but there are different total values (between " + minTotal.getAsInt() + " and " + maxTotal.getAsInt() + ").");
        }
        List sorted = segments.stream().sorted((a, b) -> Integer.compare(a.getLine(), b.getLine())).collect(Collectors.toList());
        int i = 1;
        Iterator it = sorted.iterator();
        while (it.hasNext()) {
            if (((Segment)it.next()).getLine() > i) {
                throw new InvalidLogDataException("Missing segment #" + i + " out of " + minTotal.getAsInt() + " total segments");
            }
            ++i;
        }
        return sorted.stream().map(s -> s.getContent()).collect(Collectors.joining());
    }

    private String validateHashRestore(byte[] hashed) throws InvalidLogDataException {
        byte[] digest;
        byte[] binary = Arrays.copyOfRange(hashed, 0, hashed.length - 32);
        byte[] hash = Arrays.copyOfRange(hashed, hashed.length - 32, hashed.length);
        try {
            digest = this.digest(binary);
        }
        catch (IOException e) {
            throw new InvalidLogDataException("Could compute SHA-256 digest: " + e.getMessage());
        }
        if (!Arrays.equals(hash, digest)) {
            throw new InvalidLogDataException("Corrupted data: Invalid SHA-256 digest");
        }
        return new String(binary, StandardCharsets.UTF_8);
    }

    private byte[] encodeAndHash(String data) throws IOException {
        byte[] binary = data.getBytes(StandardCharsets.UTF_8);
        byte[] digest = this.digest(binary);
        return this.concat(binary, digest);
    }

    private byte[] digest(byte[] binary) throws IOException {
        MessageDigest messageDigest;
        try {
            messageDigest = MessageDigest.getInstance(MESSAGE_DIGEST_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException(e.getMessage());
        }
        messageDigest.update(binary);
        byte[] hash = messageDigest.digest();
        return hash;
    }

    private String encodeBase64(byte[] decoded) {
        return Base64.getEncoder().encodeToString(decoded);
    }

    private byte[] decodeBase64(String encoded) throws InvalidLogDataException {
        try {
            return Base64.getDecoder().decode(encoded);
        }
        catch (Exception e) {
            throw new InvalidLogDataException("Corrupted data: Could not decode Base64: " + e.getMessage());
        }
    }

    private List<String> splitAndShield(String base64) {
        String[] lines = this.splitBySize(base64, this.segmentSize);
        String[] shielded = this.shield(lines);
        return Arrays.asList(shielded);
    }

    public static byte[] compress(byte[] input, int compressionLevel, boolean GZIPFormat) throws IOException {
        Deflater compressor = new Deflater(compressionLevel, GZIPFormat);
        compressor.setInput(input);
        compressor.finish();
        ByteArrayOutputStream bao = new ByteArrayOutputStream();
        byte[] readBuffer = new byte[1024];
        int readCount = 0;
        while (!compressor.finished()) {
            readCount = compressor.deflate(readBuffer);
            if (readCount <= 0) continue;
            bao.write(readBuffer, 0, readCount);
        }
        compressor.end();
        return bao.toByteArray();
    }

    public static byte[] decompress(byte[] input, boolean GZIPFormat) throws IOException, DataFormatException {
        Inflater decompressor = new Inflater(GZIPFormat);
        decompressor.setInput(input);
        ByteArrayOutputStream bao = new ByteArrayOutputStream();
        byte[] readBuffer = new byte[1024];
        int readCount = 0;
        while (!decompressor.finished()) {
            readCount = decompressor.inflate(readBuffer);
            if (readCount <= 0) continue;
            bao.write(readBuffer, 0, readCount);
        }
        decompressor.end();
        return bao.toByteArray();
    }

    private byte[] concat(byte[] a, byte[] b) {
        byte[] combined = new byte[a.length + b.length];
        System.arraycopy(a, 0, combined, 0, a.length);
        System.arraycopy(b, 0, combined, a.length, b.length);
        return combined;
    }

    private String[] splitBySize(String text, int size) {
        String[] split = new String[(text.length() + size - 1) / size];
        int i = 0;
        for (int start = 0; start < text.length(); start += size) {
            split[i++] = text.substring(start, Math.min(text.length(), start + size));
        }
        return split;
    }

    private String[] shield(String[] lines) {
        String[] shielded = new String[lines.length];
        for (int i = 0; i < lines.length; ++i) {
            shielded[i] = "{" + (i + 1) + "/" + lines.length + ":" + lines[i] + "}";
        }
        return shielded;
    }

    public static class InvalidLogDataException
    extends Exception {
        private static final long serialVersionUID = 1L;

        public InvalidLogDataException(String message) {
            super(message);
        }
    }

    private class Segment {
        private int line;
        private int total;
        private String content;

        public Segment(int line, int total, String content) {
            this.line = line;
            this.total = total;
            this.content = content;
        }

        public int getLine() {
            return this.line;
        }

        public int getTotal() {
            return this.total;
        }

        public String getContent() {
            return this.content;
        }

        public String toString() {
            return "{" + this.line + "/" + this.total + ":" + this.content + "}";
        }
    }
}

