package org.hotrod.torcs.decorators;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.logging.Logger;

import org.hotrod.torcs.DataSourceReference;
import org.hotrod.torcs.Torcs;

public class TorcsStatement implements Statement {

  private static final Logger log = Logger.getLogger(TorcsStatement.class.getName());

  protected Statement wrapped;
  private TorcsConnection conn;
  protected Torcs torcs;
  protected DataSourceReference dataSourceReference;

  private int batchStatementCount = 0;
  private String firstBatchSQL = null;

  public TorcsStatement(Statement wrapped, TorcsConnection conn, Torcs torcs, DataSourceReference dataSourceReference) {
    if (wrapped == null) {
      throw new RuntimeException("Cannot use a null Statement");
    }
    if (conn == null) {
      throw new RuntimeException("Cannot use a null Connection");
    }
    if (torcs == null) {
      throw new RuntimeException("Cannot use a null Torcs");
    }
    if (dataSourceReference == null) {
      throw new RuntimeException("Cannot use a null DataSourceReference");
    }
    this.wrapped = wrapped;
    this.conn = conn;
    this.torcs = torcs;
    this.dataSourceReference = dataSourceReference;
  }

  // Statement Implementation

  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException {
    return this.wrapped.isWrapperFor(iface);
  }

  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException {
    return this.wrapped.unwrap(iface);
  }

  @Override
  public void addBatch(String sql) throws SQLException {
    this.wrapped.addBatch(sql);

    if (this.batchStatementCount == 0) {
      this.firstBatchSQL = sql;
    }
    this.batchStatementCount++;

  }

  @Override
  public void cancel() throws SQLException {
    this.wrapped.cancel();
  }

  @Override
  public void clearBatch() throws SQLException {
    this.wrapped.clearBatch();
    this.batchStatementCount = 0;
    this.firstBatchSQL = null;
  }

  @Override
  public void clearWarnings() throws SQLException {
    this.wrapped.clearWarnings();
  }

  @Override
  public void close() throws SQLException {
    this.wrapped.close();
  }

  @Override
  public void closeOnCompletion() throws SQLException {
    this.wrapped.closeOnCompletion();
  }

  @Override
  public boolean execute(String sql) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      boolean r = this.wrapped.execute(sql); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      boolean r = this.wrapped.execute(sql, autoGeneratedKeys); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public boolean execute(String sql, int[] columnIndexes) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      boolean r = this.wrapped.execute(sql, columnIndexes); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public boolean execute(String sql, String[] columnNames) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      boolean r = this.wrapped.execute(sql, columnNames); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public int[] executeBatch() throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      int[] r = this.wrapped.executeBatch(); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(
          this.dataSourceReference, "-- Total of " + this.batchStatementCount
              + " batch SQL statements. First one below:\n" + coalesce(this.firstBatchSQL, ""),
          null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(
          this.dataSourceReference, "-- Total of " + this.batchStatementCount
              + " batch SQL statements. First one below:\n" + coalesce(this.firstBatchSQL, ""),
          null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public long[] executeLargeBatch() throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      long[] r = this.wrapped.executeLargeBatch(); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(
          this.dataSourceReference, "-- Total of " + this.batchStatementCount
              + " batch SQL statements. First one below:\n" + coalesce(this.firstBatchSQL, ""),
          null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(
          this.dataSourceReference, "-- Total of " + this.batchStatementCount
              + " batch SQL statements. First one below:\n" + coalesce(this.firstBatchSQL, ""),
          null, null, (int) (end - start), t);
      throw t;
    }
  }

  private String coalesce(String a, String b) {
    return a == null ? b : a;
  }

  @Override
  public ResultSet executeQuery(String sql) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      ResultSet r = this.wrapped.executeQuery(sql); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public int executeUpdate(String sql) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      int r = this.wrapped.executeUpdate(sql); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      int r = this.wrapped.executeUpdate(sql, autoGeneratedKeys); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      int r = this.wrapped.executeUpdate(sql, columnIndexes); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public int executeUpdate(String sql, String[] columnNames) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      int r = this.wrapped.executeUpdate(sql, columnNames); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public long executeLargeUpdate(String sql) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      long r = this.wrapped.executeLargeUpdate(sql); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      long r = this.wrapped.executeLargeUpdate(sql, autoGeneratedKeys); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      long r = this.wrapped.executeLargeUpdate(sql, columnIndexes); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
    log.fine("execute");
    long start = System.currentTimeMillis();
    try {
      long r = this.wrapped.executeLargeUpdate(sql, columnNames); // delegate
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), null);
      return r;
    } catch (Throwable t) {
      log.fine("executed failed");
      long end = System.currentTimeMillis();
      this.torcs.record(this.dataSourceReference, sql, null, null, (int) (end - start), t);
      throw t;
    }
  }

  @Override
  public Connection getConnection() throws SQLException {
    return this.conn;
  }

  @Override
  public int getFetchDirection() throws SQLException {
    return this.wrapped.getFetchDirection();
  }

  @Override
  public int getFetchSize() throws SQLException {
    return this.wrapped.getFetchSize();
  }

  @Override
  public ResultSet getGeneratedKeys() throws SQLException {
    return this.wrapped.getGeneratedKeys();
  }

  @Override
  public int getMaxFieldSize() throws SQLException {
    return this.wrapped.getMaxFieldSize();
  }

  @Override
  public int getMaxRows() throws SQLException {
    return this.wrapped.getMaxRows();
  }

  @Override
  public boolean getMoreResults() throws SQLException {
    return this.wrapped.getMoreResults();
  }

  @Override
  public boolean getMoreResults(int current) throws SQLException {
    return this.wrapped.getMoreResults(current);
  }

  @Override
  public int getQueryTimeout() throws SQLException {
    return this.wrapped.getQueryTimeout();
  }

  @Override
  public ResultSet getResultSet() throws SQLException {
    return this.wrapped.getResultSet();
  }

  @Override
  public int getResultSetConcurrency() throws SQLException {
    return this.wrapped.getResultSetConcurrency();
  }

  @Override
  public int getResultSetHoldability() throws SQLException {
    return this.wrapped.getResultSetHoldability();
  }

  @Override
  public int getResultSetType() throws SQLException {
    return this.wrapped.getResultSetType();
  }

  @Override
  public int getUpdateCount() throws SQLException {
    return this.wrapped.getUpdateCount();
  }

  @Override
  public SQLWarning getWarnings() throws SQLException {
    return this.wrapped.getWarnings();
  }

  @Override
  public boolean isCloseOnCompletion() throws SQLException {
    return this.wrapped.isCloseOnCompletion();
  }

  @Override
  public boolean isClosed() throws SQLException {
    return this.wrapped.isClosed();
  }

  @Override
  public boolean isPoolable() throws SQLException {
    return this.wrapped.isPoolable();
  }

  @Override
  public void setCursorName(String name) throws SQLException {
    this.wrapped.setCursorName(name);
  }

  @Override
  public void setEscapeProcessing(boolean enable) throws SQLException {
    this.wrapped.setEscapeProcessing(enable);
  }

  @Override
  public void setFetchDirection(int direction) throws SQLException {
    this.wrapped.setFetchDirection(direction);
  }

  @Override
  public void setFetchSize(int rows) throws SQLException {
    this.wrapped.setFetchSize(rows);
  }

  @Override
  public void setMaxFieldSize(int max) throws SQLException {
    this.wrapped.setMaxFieldSize(max);
  }

  @Override
  public void setMaxRows(int max) throws SQLException {
    this.wrapped.setMaxRows(max);
  }

  @Override
  public void setPoolable(boolean poolable) throws SQLException {
    this.wrapped.setPoolable(poolable);
  }

  @Override
  public void setQueryTimeout(int seconds) throws SQLException {
    this.wrapped.setQueryTimeout(seconds);
  }

}
