// Data obtained from: http://www.cdc.gov/growthcharts/percentile_data_files.htm
//                     http://www.cdc.gov/growthcharts/data/zscore/lenageinf.xls
//                     http://www.cdc.gov/growthcharts/data/zscore/statage.xls
// Fetched 4/6/2011
// JS by: Josh Mandel

/**
* Find zscore for the height of a patient
*
* @param {Object} p Parameters include height in meters, age in years, sex ('male' or 'female')
* @returns {Boolean} True if the parameter is an array
*/
var find_height_zscore = function(p) {
  var age = 12 * p.age, 
      sex = p.sex,
    height = 100 * p.height,
    exact = find_height_parameters(age, sex);

/*
From CDC:  

The LMS parameters are the median (M), the generalized coefficient of
variation (S), and the power in the Box-Cox transformation (L). To 
obtain the value (X) of a given physical measurement at a 
particular z-score or percentile, use the following equation:

To obtain the z-score (Z) and corresponding percentile for a given 
measurement (X), use the following equation:


Z = (((X/M)**L) - 1)/LS  , l10 (?)
Z = ln(X/M)/S            , L=0
*/
  if (exact === null) return 0;

  if (exact.l == 0) 
    return Math.log(height/exact.m)/exact.s;
  return (Math.pow(height/exact.m, exact.l)-1) / (exact.l * exact.s);

};

var find_height_threshold = function(p) {
  var age = 12 * p.age, 
      sex = p.sex,
      target = Math.probit(p.target / 100.0),
      exact = find_height_parameters(age, sex);
  if (exact.l == 0)
      return Math.exp(exact.s * target) * exact.m;
  return Math.pow(exact.s * exact.l * target + 1, 1.0/exact.l) * exact.m;
};

var find_height_parameters = function(age, sex) {

  function mean(a,b,weight) {
    return b * weight + a * (1-weight);
  };


  // Heuristic:  use height data preferentially
  // over length data for younger children.
  // (Since there is overlap from 24mo-36mo.)
  var ref = CDC_STATURE_DATA;
  if (age <= 24)
      ref = CDC_LENGTH_DATA;
  ref = ref[sex];

  var exact = null;
  for (var i = 0; i < ref.length-1; i++) {
    if (age == ref[i].age) {
      exact = ref[i];
      break;
    }
    else if (age > ref[i].age && age < ref[i+1].age) {
      var weight = (age-ref[i].age) / (ref[i+1].age-ref[i].age);
      var exact = { 
                    sex: sex, 
                    age: age, 
                    l:   mean(ref[i].l, ref[i+1].l, weight),
                    m:   mean(ref[i].m, ref[i+1].m, weight),
                    s:   mean(ref[i].s, ref[i+1].s, weight)
                 };
      break;
    }
  }
  return exact;
}


var CDC_LENGTH_DATA = {
'male': [
{'sex': 'male', 'age': 0,  'l': 1.2670042261, 'm': 49.988884079, 's': 0.0531121908},
{'sex': 'male', 'age': 0.5,  'l': 0.5112376962, 'm': 52.695975301, 's': 0.0486926838},
{'sex': 'male', 'age': 1.5,  'l': -0.45224446, 'm': 56.628428552, 's': 0.0441168302},
{'sex': 'male', 'age': 2.5,  'l': -0.990594599, 'm': 59.608953427, 's': 0.0417955825},
{'sex': 'male', 'age': 3.5,  'l': -1.285837689, 'm': 62.077000266, 's': 0.0404541256},
{'sex': 'male', 'age': 4.5,  'l': -1.43031238, 'm': 64.216864104, 's': 0.0396338789},
{'sex': 'male', 'age': 5.5,  'l': -1.47657547, 'm': 66.125314898, 's': 0.0391238128},
{'sex': 'male', 'age': 6.5,  'l': -1.456837849, 'm': 67.860179904, 's': 0.0388119944},
{'sex': 'male', 'age': 7.5,  'l': -1.391898768, 'm': 69.459084582, 's': 0.0386332091},
{'sex': 'male', 'age': 8.5,  'l': -1.29571459, 'm': 70.948039123, 's': 0.0385468328},
{'sex': 'male', 'age': 9.5,  'l': -1.177919048, 'm': 72.345861109, 's': 0.0385262623},
{'sex': 'male', 'age': 10.5,  'l': -1.045326049, 'm': 73.666654103, 's': 0.038553387},
{'sex': 'male', 'age': 11.5,  'l': -0.902800887, 'm': 74.921297174, 's': 0.0386155012},
{'sex': 'male', 'age': 12.5,  'l': -0.753908107, 'm': 76.118375358, 's': 0.0387034611},
{'sex': 'male', 'age': 13.5,  'l': -0.601263523, 'm': 77.264799111, 's': 0.0388105571},
{'sex': 'male', 'age': 14.5,  'l': -0.446805039, 'm': 78.366223087, 's': 0.0389317838},
{'sex': 'male', 'age': 15.5,  'l': -0.291974772, 'm': 79.427340501, 's': 0.0390633563},
{'sex': 'male', 'age': 16.5,  'l': -0.13784767, 'm': 80.452094919, 's': 0.0392023816},
{'sex': 'male', 'age': 17.5,  'l': 0.014776155, 'm': 81.443836034, 's': 0.0393466285},
{'sex': 'male', 'age': 18.5,  'l': 0.1653041691, 'm': 82.405436434, 's': 0.0394943647},
{'sex': 'male', 'age': 19.5,  'l': 0.3133018086, 'm': 83.339380627, 's': 0.0396442379},
{'sex': 'male', 'age': 20.5,  'l': 0.4584554707, 'm': 84.247833944, 's': 0.0397951891},
{'sex': 'male', 'age': 21.5,  'l': 0.6005446308, 'm': 85.132696575, 's': 0.0399463877},
{'sex': 'male', 'age': 22.5,  'l': 0.7394389526, 'm': 85.995648803, 's': 0.0400971806},
{'sex': 'male', 'age': 23.5,  'l': 0.8750004465, 'm': 86.838175097, 's': 0.0402470597},
{'sex': 'male', 'age': 24.5,  'l': 1.0072080698, 'm': 87.661609339, 's': 0.0403956257},
{'sex': 'male', 'age': 25.5,  'l': 0.8372513507, 'm': 88.452472817, 's': 0.0405775247},
{'sex': 'male', 'age': 26.5,  'l': 0.681492975, 'm': 89.223264339, 's': 0.0407231216},
{'sex': 'male', 'age': 27.5,  'l': 0.5387796541, 'm': 89.975492278, 's': 0.0408331942},
{'sex': 'male', 'age': 28.5,  'l': 0.4076971531, 'm': 90.710408526, 's': 0.0409090589},
{'sex': 'male', 'age': 29.5,  'l': 0.2867624533, 'm': 91.429077621, 's': 0.0409524329},
{'sex': 'male', 'age': 30.5,  'l': 0.1744894848, 'm': 92.132423789, 's': 0.0409653301},
{'sex': 'male', 'age': 31.5,  'l': 0.0694445205, 'm': 92.821271673, 's': 0.0409499758},
{'sex': 'male', 'age': 32.5,  'l': -0.029720564, 'm': 93.496379457, 's': 0.0409087367},
{'sex': 'male', 'age': 33.5,  'l': -0.124251789, 'm': 94.158465459, 's': 0.0408440616},
{'sex': 'male', 'age': 34.5,  'l': -0.215288396, 'm': 94.808229231, 's': 0.0407584307},
{'sex': 'male', 'age': 35.5,  'l': -0.30385434, 'm': 95.446369813, 's': 0.0406543122}],

'female': [
{'sex': 'female', 'age': 0,  'l': -1.295960857, 'm': 49.286396118, 's': 0.0500855601},
{'sex': 'female', 'age': 0.5,  'l': -0.809249882, 'm': 51.683580573, 's': 0.0468185454},
{'sex': 'female', 'age': 1.5,  'l': -0.050782985, 'm': 55.286128126, 's': 0.0434439},
{'sex': 'female', 'age': 2.5,  'l': 0.4768514065, 'm': 58.093819061, 's': 0.0417161032},
{'sex': 'female', 'age': 3.5,  'l': 0.8432996117, 'm': 60.459807634, 's': 0.0407051733},
{'sex': 'female', 'age': 4.5,  'l': 1.0975622571, 'm': 62.536696555, 's': 0.0400797646},
{'sex': 'female', 'age': 5.5,  'l': 1.2725096408, 'm': 64.406327624, 's': 0.0396868449},
{'sex': 'female', 'age': 6.5,  'l': 1.3904288587, 'm': 66.118415533, 's': 0.0394445547},
{'sex': 'female', 'age': 7.5,  'l': 1.466733925, 'm': 67.705744192, 's': 0.0393047376},
{'sex': 'female', 'age': 8.5,  'l': 1.5123019758, 'm': 69.191236138, 's': 0.0392371101},
{'sex': 'female', 'age': 9.5,  'l': 1.534950767, 'm': 70.591639237, 's': 0.0392216648},
{'sex': 'female', 'age': 10.5,  'l': 1.5403908751, 'm': 71.919616727, 's': 0.0392446716},
{'sex': 'female', 'age': 11.5,  'l': 1.5328528917, 'm': 73.185010399, 's': 0.0392964203},
{'sex': 'female', 'age': 12.5,  'l': 1.5155094695, 'm': 74.395643786, 's': 0.0393698746},
{'sex': 'female', 'age': 13.5,  'l': 1.4907650275, 'm': 75.557854397, 's': 0.0394598321},
{'sex': 'female', 'age': 14.5,  'l': 1.460458255, 'm': 76.676858713, 's': 0.0395623818},
{'sex': 'female', 'age': 15.5,  'l': 1.4260060091, 'm': 77.757009856, 's': 0.0396745415},
{'sex': 'female', 'age': 16.5,  'l': 1.3885070954, 'm': 78.801984056, 's': 0.0397940102},
{'sex': 'female', 'age': 17.5,  'l': 1.3488181274, 'm': 79.814918523, 's': 0.0399189943},
{'sex': 'female', 'age': 18.5,  'l': 1.3076096543, 'm': 80.798515316, 's': 0.0400480838},
{'sex': 'female', 'age': 19.5,  'l': 1.2654081486, 'm': 81.755120921, 's': 0.0401801621},
{'sex': 'female', 'age': 20.5,  'l': 1.2226277319, 'm': 82.686788098, 's': 0.0403143396},
{'sex': 'female', 'age': 21.5,  'l': 1.1795943654, 'm': 83.59532461, 's': 0.040449904},
{'sex': 'female', 'age': 22.5,  'l': 1.1365644483, 'm': 84.48233206, 's': 0.0405862829},
{'sex': 'female', 'age': 23.5,  'l': 1.0937319466, 'm': 85.349236238, 's': 0.0407230154},
{'sex': 'female', 'age': 24.5,  'l': 1.0512729122, 'm': 86.197316902, 's': 0.0408597268},
{'sex': 'female', 'age': 25.5,  'l': 1.041951175, 'm': 87.090263184, 's': 0.0411421607},
{'sex': 'female', 'age': 26.5,  'l': 1.0125922357, 'm': 87.957141816, 's': 0.0413493986},
{'sex': 'female', 'age': 27.5,  'l': 0.9705419086, 'm': 88.796018399, 's': 0.041500428},
{'sex': 'female', 'age': 28.5,  'l': 0.9211299875, 'm': 89.605511504, 's': 0.0416105082},
{'sex': 'female', 'age': 29.5,  'l': 0.8682213922, 'm': 90.38476689, 's': 0.0416917614},
{'sex': 'female', 'age': 30.5,  'l': 0.8145441296, 'm': 91.133417221, 's': 0.0417536804},
{'sex': 'female', 'age': 31.5,  'l': 0.7619579771, 'm': 91.851543603, 's': 0.0418035615},
{'sex': 'female', 'age': 32.5,  'l': 0.7116602284, 'm': 92.539635196, 's': 0.0418468815},
{'sex': 'female', 'age': 33.5,  'l': 0.6643233788, 'm': 93.198544285, 's': 0.041887626},
{'sex': 'female', 'age': 34.5,  'l': 0.6202851017, 'm': 93.829453915, 's': 0.0419285678},
{'sex': 'female', 'age': 35.5,  'l': 0.57955631, 'm': 94.433822779, 's': 0.0419715144},
]};

var CDC_STATURE_DATA = {
'male': [
{'sex': 'male', 'age': 24,  'l': 0.941523967, 'm': 86.45220101, 's': 0.040321528},
{'sex': 'male', 'age': 24.5,  'l': 1.00720807, 'm': 86.86160934, 's': 0.040395626},
{'sex': 'male', 'age': 25.5,  'l': 0.837251351, 'm': 87.65247282, 's': 0.040577525},
{'sex': 'male', 'age': 26.5,  'l': 0.681492975, 'm': 88.42326434, 's': 0.040723122},
{'sex': 'male', 'age': 27.5,  'l': 0.538779654, 'm': 89.17549228, 's': 0.040833194},
{'sex': 'male', 'age': 28.5,  'l': 0.407697153, 'm': 89.91040853, 's': 0.040909059},
{'sex': 'male', 'age': 29.5,  'l': 0.286762453, 'm': 90.62907762, 's': 0.040952433},
{'sex': 'male', 'age': 30.5,  'l': 0.174489485, 'm': 91.33242379, 's': 0.04096533},
{'sex': 'male', 'age': 31.5,  'l': 0.069444521, 'm': 92.02127167, 's': 0.040949976},
{'sex': 'male', 'age': 32.5,  'l': -0.029720564, 'm': 92.69637946, 's': 0.040908737},
{'sex': 'male', 'age': 33.5,  'l': -0.124251789, 'm': 93.35846546, 's': 0.040844062},
{'sex': 'male', 'age': 34.5,  'l': -0.215288396, 'm': 94.00822923, 's': 0.040758431},
{'sex': 'male', 'age': 35.5,  'l': -0.30385434, 'm': 94.64636981, 's': 0.040654312},
{'sex': 'male', 'age': 36.5,  'l': -0.390918369, 'm': 95.27359106, 's': 0.04053412},
{'sex': 'male', 'age': 37.5,  'l': -0.254801167, 'm': 95.91474929, 's': 0.040572876},
{'sex': 'male', 'age': 38.5,  'l': -0.125654535, 'm': 96.54734328, 's': 0.04061691},
{'sex': 'male', 'age': 39.5,  'l': -0.00316735, 'm': 97.17191309, 's': 0.040666414},
{'sex': 'male', 'age': 40.5,  'l': 0.11291221, 'm': 97.78897727, 's': 0.040721467},
{'sex': 'male', 'age': 41.5,  'l': 0.222754969, 'm': 98.3990283, 's': 0.040782045},
{'sex': 'male', 'age': 42.5,  'l': 0.326530126, 'm': 99.00254338, 's': 0.040848042},
{'sex': 'male', 'age': 43.5,  'l': 0.42436156, 'm': 99.599977, 's': 0.040919281},
{'sex': 'male', 'age': 44.5,  'l': 0.516353108, 'm': 100.191764, 's': 0.040995524},
{'sex': 'male', 'age': 45.5,  'l': 0.602595306, 'm': 100.7783198, 's': 0.041076485},
{'sex': 'male', 'age': 46.5,  'l': 0.683170764, 'm': 101.3600411, 's': 0.041161838},
{'sex': 'male', 'age': 47.5,  'l': 0.758158406, 'm': 101.9373058, 's': 0.041251224},
{'sex': 'male', 'age': 48.5,  'l': 0.827636736, 'm': 102.5104735, 's': 0.041344257},
{'sex': 'male', 'age': 49.5,  'l': 0.891686306, 'm': 103.0798852, 's': 0.041440534},
{'sex': 'male', 'age': 50.5,  'l': 0.95039153, 'm': 103.645864, 's': 0.041539635},
{'sex': 'male', 'age': 51.5,  'l': 1.003830006, 'm': 104.208713, 's': 0.041641136},
{'sex': 'male', 'age': 52.5,  'l': 1.05213569, 'm': 104.7687256, 's': 0.041744602},
{'sex': 'male', 'age': 53.5,  'l': 1.0953669, 'm': 105.3261638, 's': 0.041849607},
{'sex': 'male', 'age': 54.5,  'l': 1.133652119, 'm': 105.8812823, 's': 0.041955723},
{'sex': 'male', 'age': 55.5,  'l': 1.167104213, 'm': 106.4343146, 's': 0.042062532},
{'sex': 'male', 'age': 56.5,  'l': 1.195845353, 'm': 106.9854769, 's': 0.042169628},
{'sex': 'male', 'age': 57.5,  'l': 1.220004233, 'm': 107.534968, 's': 0.042276619},
{'sex': 'male', 'age': 58.5,  'l': 1.239715856, 'm': 108.0829695, 's': 0.042383129},
{'sex': 'male', 'age': 59.5,  'l': 1.255121285, 'm': 108.6296457, 's': 0.042488804},
{'sex': 'male', 'age': 60.5,  'l': 1.266367398, 'm': 109.1751441, 's': 0.042593311},
{'sex': 'male', 'age': 61.5,  'l': 1.273606657, 'm': 109.7195954, 's': 0.042696342},
{'sex': 'male', 'age': 62.5,  'l': 1.276996893, 'm': 110.2631136, 's': 0.042797615},
{'sex': 'male', 'age': 63.5,  'l': 1.276701119, 'm': 110.8057967, 's': 0.042896877},
{'sex': 'male', 'age': 64.5,  'l': 1.272887366, 'm': 111.3477265, 's': 0.042993904},
{'sex': 'male', 'age': 65.5,  'l': 1.265728536, 'm': 111.8889694, 's': 0.043088503},
{'sex': 'male', 'age': 66.5,  'l': 1.255402281, 'm': 112.4295761, 's': 0.043180513},
{'sex': 'male', 'age': 67.5,  'l': 1.242090871, 'm': 112.9695827, 's': 0.043269806},
{'sex': 'male', 'age': 68.5,  'l': 1.225981067, 'm': 113.5090108, 's': 0.043356287},
{'sex': 'male', 'age': 69.5,  'l': 1.207263978, 'm': 114.0478678, 's': 0.043439893},
{'sex': 'male', 'age': 70.5,  'l': 1.186140222, 'm': 114.5861486, 's': 0.043520597},
{'sex': 'male', 'age': 71.5,  'l': 1.162796198, 'm': 115.1238315, 's': 0.043598407},
{'sex': 'male', 'age': 72.5,  'l': 1.137442868, 'm': 115.6608862, 's': 0.043673359},
{'sex': 'male', 'age': 73.5,  'l': 1.110286487, 'm': 116.1972691, 's': 0.043745523},
{'sex': 'male', 'age': 74.5,  'l': 1.081536236, 'm': 116.732925, 's': 0.043815003},
{'sex': 'male', 'age': 75.5,  'l': 1.05140374, 'm': 117.2677879, 's': 0.043881929},
{'sex': 'male', 'age': 76.5,  'l': 1.020102497, 'm': 117.8017819, 's': 0.043946461},
{'sex': 'male', 'age': 77.5,  'l': 0.987847213, 'm': 118.3348215, 's': 0.044008785},
{'sex': 'male', 'age': 78.5,  'l': 0.954853043, 'm': 118.8668123, 's': 0.044069112},
{'sex': 'male', 'age': 79.5,  'l': 0.921334742, 'm': 119.397652, 's': 0.044127675},
{'sex': 'male', 'age': 80.5,  'l': 0.887505723, 'm': 119.9272309, 's': 0.044184725},
{'sex': 'male', 'age': 81.5,  'l': 0.85357703, 'm': 120.455433, 's': 0.044240532},
{'sex': 'male', 'age': 82.5,  'l': 0.819756239, 'm': 120.9821362, 's': 0.044295379},
{'sex': 'male', 'age': 83.5,  'l': 0.786246296, 'm': 121.5072136, 's': 0.044349559},
{'sex': 'male', 'age': 84.5,  'l': 0.753244292, 'm': 122.0305342, 's': 0.044403374},
{'sex': 'male', 'age': 85.5,  'l': 0.720940222, 'm': 122.5519634, 's': 0.04445713},
{'sex': 'male', 'age': 86.5,  'l': 0.689515708, 'm': 123.0713645, 's': 0.044511135},
{'sex': 'male', 'age': 87.5,  'l': 0.659142731, 'm': 123.588599, 's': 0.044565693},
{'sex': 'male', 'age': 88.5,  'l': 0.629997853, 'm': 124.1035312, 's': 0.044621104},
{'sex': 'male', 'age': 89.5,  'l': 0.602203984, 'm': 124.6160161, 's': 0.044677662},
{'sex': 'male', 'age': 90.5,  'l': 0.575908038, 'm': 125.1259182, 's': 0.044735646},
{'sex': 'male', 'age': 91.5,  'l': 0.55123134, 'm': 125.6331012, 's': 0.044795322},
{'sex': 'male', 'age': 92.5,  'l': 0.528279901, 'm': 126.1374319, 's': 0.044856941},
{'sex': 'male', 'age': 93.5,  'l': 0.507143576, 'm': 126.6387804, 's': 0.04492073},
{'sex': 'male', 'age': 94.5,  'l': 0.487895344, 'm': 127.1370217, 's': 0.044986899},
{'sex': 'male', 'age': 95.5,  'l': 0.470590753, 'm': 127.6320362, 's': 0.045055632},
{'sex': 'male', 'age': 96.5,  'l': 0.455267507, 'm': 128.1237104, 's': 0.045127088},
{'sex': 'male', 'age': 97.5,  'l': 0.441945241, 'm': 128.6119383, 's': 0.045201399},
{'sex': 'male', 'age': 98.5,  'l': 0.430625458, 'm': 129.096622, 's': 0.045278671},
{'sex': 'male', 'age': 99.5,  'l': 0.421291648, 'm': 129.5776723, 's': 0.045358979},
{'sex': 'male', 'age': 100.5,  'l': 0.413909588, 'm': 130.0550101, 's': 0.045442372},
{'sex': 'male', 'age': 101.5,  'l': 0.408427813, 'm': 130.5285669, 's': 0.045528869},
{'sex': 'male', 'age': 102.5,  'l': 0.404778262, 'm': 130.9982857, 's': 0.045618459},
{'sex': 'male', 'age': 103.5,  'l': 0.402877077, 'm': 131.4641218, 's': 0.045711105},
{'sex': 'male', 'age': 104.5,  'l': 0.402625561, 'm': 131.9260439, 's': 0.045806742},
{'sex': 'male', 'age': 105.5,  'l': 0.40391127, 'm': 132.3840348, 's': 0.045905281},
{'sex': 'male', 'age': 106.5,  'l': 0.406609232, 'm': 132.838092, 's': 0.046006604},
{'sex': 'male', 'age': 107.5,  'l': 0.410583274, 'm': 133.2882291, 's': 0.046110573},
{'sex': 'male', 'age': 108.5,  'l': 0.415687443, 'm': 133.7344759, 's': 0.046217028},
{'sex': 'male', 'age': 109.5,  'l': 0.421767514, 'm': 134.1768801, 's': 0.04632579},
{'sex': 'male', 'age': 110.5,  'l': 0.428662551, 'm': 134.6155076, 's': 0.046436662},
{'sex': 'male', 'age': 111.5,  'l': 0.436206531, 'm': 135.0504433, 's': 0.04654943},
{'sex': 'male', 'age': 112.5,  'l': 0.44423, 'm': 135.4817925, 's': 0.046663871},
{'sex': 'male', 'age': 113.5,  'l': 0.45256176, 'm': 135.9096813, 's': 0.046779748},
{'sex': 'male', 'age': 114.5,  'l': 0.461030578, 'm': 136.3342577, 's': 0.046896817},
{'sex': 'male', 'age': 115.5,  'l': 0.469466904, 'm': 136.7556923, 's': 0.047014827},
{'sex': 'male', 'age': 116.5,  'l': 0.477704608, 'm': 137.1741794, 's': 0.047133525},
{'sex': 'male', 'age': 117.5,  'l': 0.48558272, 'm': 137.5899378, 's': 0.047252654},
{'sex': 'male', 'age': 118.5,  'l': 0.492947182, 'm': 138.0032114, 's': 0.047371961},
{'sex': 'male', 'age': 119.5,  'l': 0.499652617, 'm': 138.4142703, 's': 0.047491194},
{'sex': 'male', 'age': 120.5,  'l': 0.505564115, 'm': 138.8234114, 's': 0.047610108},
{'sex': 'male', 'age': 121.5,  'l': 0.510559047, 'm': 139.2309592, 's': 0.047728463},
{'sex': 'male', 'age': 122.5,  'l': 0.514528903, 'm': 139.6372663, 's': 0.04784603},
{'sex': 'male', 'age': 123.5,  'l': 0.517381177, 'm': 140.042714, 's': 0.047962592},
{'sex': 'male', 'age': 124.5,  'l': 0.519041285, 'm': 140.4477127, 's': 0.048077942},
{'sex': 'male', 'age': 125.5,  'l': 0.519454524, 'm': 140.8527022, 's': 0.048191889},
{'sex': 'male', 'age': 126.5,  'l': 0.518588072, 'm': 141.2581515, 's': 0.048304259},
{'sex': 'male', 'age': 127.5,  'l': 0.516433004, 'm': 141.6645592, 's': 0.048414893},
{'sex': 'male', 'age': 128.5,  'l': 0.513006312, 'm': 142.072452, 's': 0.048523648},
{'sex': 'male', 'age': 129.5,  'l': 0.508352901, 'm': 142.4823852, 's': 0.048630402},
{'sex': 'male', 'age': 130.5,  'l': 0.502547502, 'm': 142.8949403, 's': 0.04873505},
{'sex': 'male', 'age': 131.5,  'l': 0.495696454, 'm': 143.3107241, 's': 0.048837504},
{'sex': 'male', 'age': 132.5,  'l': 0.487939275, 'm': 143.7303663, 's': 0.048937694},
{'sex': 'male', 'age': 133.5,  'l': 0.479449924, 'm': 144.1545167, 's': 0.049035564},
{'sex': 'male', 'age': 134.5,  'l': 0.470437652, 'm': 144.5838414, 's': 0.049131073},
{'sex': 'male', 'age': 135.5,  'l': 0.461147305, 'm': 145.0190192, 's': 0.049224189},
{'sex': 'male', 'age': 136.5,  'l': 0.451858946, 'm': 145.4607359, 's': 0.049314887},
{'sex': 'male', 'age': 137.5,  'l': 0.442886661, 'm': 145.9096784, 's': 0.049403145},
{'sex': 'male', 'age': 138.5,  'l': 0.434576385, 'm': 146.3665278, 's': 0.049488934},
{'sex': 'male', 'age': 139.5,  'l': 0.427302633, 'm': 146.8319513, 's': 0.049572216},
{'sex': 'male', 'age': 140.5,  'l': 0.421464027, 'm': 147.3065929, 's': 0.049652935},
{'sex': 'male', 'age': 141.5,  'l': 0.417477538, 'm': 147.7910635, 's': 0.049731004},
{'sex': 'male', 'age': 142.5,  'l': 0.415771438, 'm': 148.2859294, 's': 0.0498063},
{'sex': 'male', 'age': 143.5,  'l': 0.416777012, 'm': 148.7917006, 's': 0.04987865},
{'sex': 'male', 'age': 144.5,  'l': 0.420919142, 'm': 149.3088178, 's': 0.049947823},
{'sex': 'male', 'age': 145.5,  'l': 0.428606007, 'm': 149.8376391, 's': 0.050013518},
{'sex': 'male', 'age': 146.5,  'l': 0.440218167, 'm': 150.3784267, 's': 0.050075353},
{'sex': 'male', 'age': 147.5,  'l': 0.456097443, 'm': 150.9313331, 's': 0.050132858},
{'sex': 'male', 'age': 148.5,  'l': 0.476536014, 'm': 151.4963887, 's': 0.050185471},
{'sex': 'male', 'age': 149.5,  'l': 0.501766234, 'm': 152.0734897, 's': 0.050232532},
{'sex': 'male', 'age': 150.5,  'l': 0.531951655, 'm': 152.6623878, 's': 0.050273285},
{'sex': 'male', 'age': 151.5,  'l': 0.567179725, 'm': 153.2626819, 's': 0.050306885},
{'sex': 'male', 'age': 152.5,  'l': 0.607456565, 'm': 153.8738124, 's': 0.050332406},
{'sex': 'male', 'age': 153.5,  'l': 0.652704121, 'm': 154.495058, 's': 0.05034886},
{'sex': 'male', 'age': 154.5,  'l': 0.702759868, 'm': 155.1255365, 's': 0.050355216},
{'sex': 'male', 'age': 155.5,  'l': 0.757379106, 'm': 155.7642086, 's': 0.050350423},
{'sex': 'male', 'age': 156.5,  'l': 0.816239713, 'm': 156.4098858, 's': 0.050333444},
{'sex': 'male', 'age': 157.5,  'l': 0.878947416, 'm': 157.0612415, 's': 0.050303283},
{'sex': 'male', 'age': 158.5,  'l': 0.945053486, 'm': 157.7168289, 's': 0.050259018},
{'sex': 'male', 'age': 159.5,  'l': 1.014046108, 'm': 158.3750929, 's': 0.050199837},
{'sex': 'male', 'age': 160.5,  'l': 1.085383319, 'm': 159.034399, 's': 0.050125062},
{'sex': 'male', 'age': 161.5,  'l': 1.158487278, 'm': 159.6930501, 's': 0.05003418},
{'sex': 'male', 'age': 162.5,  'l': 1.232768816, 'm': 160.3493168, 's': 0.049926861},
{'sex': 'male', 'age': 163.5,  'l': 1.307628899, 'm': 161.0014586, 's': 0.049802977},
{'sex': 'male', 'age': 164.5,  'l': 1.382473225, 'm': 161.6477515, 's': 0.04966261},
{'sex': 'male', 'age': 165.5,  'l': 1.456720479, 'm': 162.2865119, 's': 0.049506051},
{'sex': 'male', 'age': 166.5,  'l': 1.529810247, 'm': 162.9161202, 's': 0.049333801},
{'sex': 'male', 'age': 167.5,  'l': 1.601219573, 'm': 163.535045, 's': 0.049146553},
{'sex': 'male', 'age': 168.5,  'l': 1.670433444, 'm': 164.1418486, 's': 0.04894519},
{'sex': 'male', 'age': 169.5,  'l': 1.736995571, 'm': 164.7352199, 's': 0.048730749},
{'sex': 'male', 'age': 170.5,  'l': 1.800483802, 'm': 165.3139755, 's': 0.048504404},
{'sex': 'male', 'age': 171.5,  'l': 1.860518777, 'm': 165.8770715, 's': 0.048267442},
{'sex': 'male', 'age': 172.5,  'l': 1.916765525, 'm': 166.4236087, 's': 0.04802123},
{'sex': 'male', 'age': 173.5,  'l': 1.968934444, 'm': 166.9528354, 's': 0.047767192},
{'sex': 'male', 'age': 174.5,  'l': 2.016781776, 'm': 167.4641466, 's': 0.047506783},
{'sex': 'male', 'age': 175.5,  'l': 2.060109658, 'm': 167.9570814, 's': 0.047241456},
{'sex': 'male', 'age': 176.5,  'l': 2.098765817, 'm': 168.4313175, 's': 0.04697265},
{'sex': 'male', 'age': 177.5,  'l': 2.132642948, 'm': 168.8866644, 's': 0.046701759},
{'sex': 'male', 'age': 178.5,  'l': 2.16167779, 'm': 169.3230548, 's': 0.046430122},
{'sex': 'male', 'age': 179.5,  'l': 2.185849904, 'm': 169.7405351, 's': 0.046159004},
{'sex': 'male', 'age': 180.5,  'l': 2.205180153, 'm': 170.139255, 's': 0.045889585},
{'sex': 'male', 'age': 181.5,  'l': 2.219728869, 'm': 170.5194567, 's': 0.045622955},
{'sex': 'male', 'age': 182.5,  'l': 2.2295937, 'm': 170.881464, 's': 0.045360101},
{'sex': 'male', 'age': 183.5,  'l': 2.234907144, 'm': 171.2256717, 's': 0.045101913},
{'sex': 'male', 'age': 184.5,  'l': 2.235833767, 'm': 171.5525345, 's': 0.044849174},
{'sex': 'male', 'age': 185.5,  'l': 2.232567138, 'm': 171.8625576, 's': 0.044602566},
{'sex': 'male', 'age': 186.5,  'l': 2.2253265, 'm': 172.1562865, 's': 0.044362674},
{'sex': 'male', 'age': 187.5,  'l': 2.214353232, 'm': 172.4342983, 's': 0.044129985},
{'sex': 'male', 'age': 188.5,  'l': 2.199905902, 'm': 172.6971935, 's': 0.043904897},
{'sex': 'male', 'age': 189.5,  'l': 2.182262864, 'm': 172.9455898, 's': 0.043687723},
{'sex': 'male', 'age': 190.5,  'l': 2.161704969, 'm': 173.180112, 's': 0.043478698},
{'sex': 'male', 'age': 191.5,  'l': 2.138524662, 'm': 173.4013896, 's': 0.043277987},
{'sex': 'male', 'age': 192.5,  'l': 2.113023423, 'm': 173.6100518, 's': 0.043085685},
{'sex': 'male', 'age': 193.5,  'l': 2.085490286, 'm': 173.8067179, 's': 0.042901835},
{'sex': 'male', 'age': 194.5,  'l': 2.0562195, 'm': 173.9919998, 's': 0.042726424},
{'sex': 'male', 'age': 195.5,  'l': 2.025496648, 'm': 174.1664951, 's': 0.042559396},
{'sex': 'male', 'age': 196.5,  'l': 1.993598182, 'm': 174.3307855, 's': 0.042400652},
{'sex': 'male', 'age': 197.5,  'l': 1.960789092, 'm': 174.4854344, 's': 0.042250063},
{'sex': 'male', 'age': 198.5,  'l': 1.927320937, 'm': 174.6309856, 's': 0.042107465},
{'sex': 'male', 'age': 199.5,  'l': 1.89343024, 'm': 174.7679617, 's': 0.041972676},
{'sex': 'male', 'age': 200.5,  'l': 1.859337259, 'm': 174.8968634, 's': 0.041845488},
{'sex': 'male', 'age': 201.5,  'l': 1.825245107, 'm': 175.0181691, 's': 0.041725679},
{'sex': 'male', 'age': 202.5,  'l': 1.791339209, 'm': 175.1323345, 's': 0.041613015},
{'sex': 'male', 'age': 203.5,  'l': 1.757787065, 'm': 175.2397926, 's': 0.041507249},
{'sex': 'male', 'age': 204.5,  'l': 1.724738292, 'm': 175.340954, 's': 0.041408129},
{'sex': 'male', 'age': 205.5,  'l': 1.692324905, 'm': 175.4362071, 's': 0.041315398},
{'sex': 'male', 'age': 206.5,  'l': 1.660661815, 'm': 175.5259191, 's': 0.041228796},
{'sex': 'male', 'age': 207.5,  'l': 1.629847495, 'm': 175.6104358, 's': 0.04114806},
{'sex': 'male', 'age': 208.5,  'l': 1.599964788, 'm': 175.690083, 's': 0.041072931},
{'sex': 'male', 'age': 209.5,  'l': 1.571081817, 'm': 175.7651671, 's': 0.04100315},
{'sex': 'male', 'age': 210.5,  'l': 1.543252982, 'm': 175.8359757, 's': 0.040938463},
{'sex': 'male', 'age': 211.5,  'l': 1.516519998, 'm': 175.9027788, 's': 0.040878617},
{'sex': 'male', 'age': 212.5,  'l': 1.490912963, 'm': 175.9658293, 's': 0.040823368},
{'sex': 'male', 'age': 213.5,  'l': 1.466451429, 'm': 176.0253641, 's': 0.040772475},
{'sex': 'male', 'age': 214.5,  'l': 1.44314546, 'm': 176.081605, 's': 0.040725706},
{'sex': 'male', 'age': 215.5,  'l': 1.420996665, 'm': 176.1347593, 's': 0.040682834},
{'sex': 'male', 'age': 216.5,  'l': 1.399999187, 'm': 176.1850208, 's': 0.04064364},
{'sex': 'male', 'age': 217.5,  'l': 1.380140651, 'm': 176.2325707, 's': 0.040607913},
{'sex': 'male', 'age': 218.5,  'l': 1.361403047, 'm': 176.2775781, 's': 0.040575448},
{'sex': 'male', 'age': 219.5,  'l': 1.343763564, 'm': 176.3202008, 's': 0.040546051},
{'sex': 'male', 'age': 220.5,  'l': 1.327195355, 'm': 176.3605864, 's': 0.040519532},
{'sex': 'male', 'age': 221.5,  'l': 1.311668242, 'm': 176.3988725, 's': 0.040495713},
{'sex': 'male', 'age': 222.5,  'l': 1.297149359, 'm': 176.4351874, 's': 0.040474421},
{'sex': 'male', 'age': 223.5,  'l': 1.283603728, 'm': 176.469651, 's': 0.040455493},
{'sex': 'male', 'age': 224.5,  'l': 1.270994782, 'm': 176.5023751, 's': 0.040438773},
{'sex': 'male', 'age': 225.5,  'l': 1.25928483, 'm': 176.533464, 's': 0.040424111},
{'sex': 'male', 'age': 226.5,  'l': 1.248435461, 'm': 176.5630153, 's': 0.040411366},
{'sex': 'male', 'age': 227.5,  'l': 1.23840791, 'm': 176.5911197, 's': 0.040400405},
{'sex': 'male', 'age': 228.5,  'l': 1.229163362, 'm': 176.6178621, 's': 0.040391101},
{'sex': 'male', 'age': 229.5,  'l': 1.220663228, 'm': 176.6433219, 's': 0.040383334},
{'sex': 'male', 'age': 230.5,  'l': 1.212869374, 'm': 176.6675729, 's': 0.04037699},
{'sex': 'male', 'age': 231.5,  'l': 1.20574431, 'm': 176.6906844, 's': 0.040371962},
{'sex': 'male', 'age': 232.5,  'l': 1.199251356, 'm': 176.712721, 's': 0.040368149},
{'sex': 'male', 'age': 233.5,  'l': 1.19335477, 'm': 176.733743, 's': 0.040365456},
{'sex': 'male', 'age': 234.5,  'l': 1.188019859, 'm': 176.753807, 's': 0.040363795},
{'sex': 'male', 'age': 235.5,  'l': 1.183213059, 'm': 176.7729657, 's': 0.04036308},
{'sex': 'male', 'age': 236.5,  'l': 1.178901998, 'm': 176.7912687, 's': 0.040363233},
{'sex': 'male', 'age': 237.5,  'l': 1.175055543, 'm': 176.8087622, 's': 0.040364179},
{'sex': 'male', 'age': 238.5,  'l': 1.171643828, 'm': 176.8254895, 's': 0.04036585},
{'sex': 'male', 'age': 239.5,  'l': 1.16863827, 'm': 176.8414914, 's': 0.04036818},
{'sex': 'male', 'age': 240,  'l': 1.167279219, 'm': 176.8492322, 's': 0.040369574}],
'female': [
{'sex': 'female', 'age': 24,  'l': 1.07244896, 'm': 84.97555512, 's': 0.040791394},
{'sex': 'female', 'age': 24.5,  'l': 1.051272912, 'm': 85.3973169, 's': 0.040859727},
{'sex': 'female', 'age': 25.5,  'l': 1.041951175, 'm': 86.29026318, 's': 0.041142161},
{'sex': 'female', 'age': 26.5,  'l': 1.012592236, 'm': 87.15714182, 's': 0.041349399},
{'sex': 'female', 'age': 27.5,  'l': 0.970541909, 'm': 87.9960184, 's': 0.041500428},
{'sex': 'female', 'age': 28.5,  'l': 0.921129988, 'm': 88.8055115, 's': 0.041610508},
{'sex': 'female', 'age': 29.5,  'l': 0.868221392, 'm': 89.58476689, 's': 0.041691761},
{'sex': 'female', 'age': 30.5,  'l': 0.81454413, 'm': 90.33341722, 's': 0.04175368},
{'sex': 'female', 'age': 31.5,  'l': 0.761957977, 'm': 91.0515436, 's': 0.041803562},
{'sex': 'female', 'age': 32.5,  'l': 0.711660228, 'm': 91.7396352, 's': 0.041846882},
{'sex': 'female', 'age': 33.5,  'l': 0.664323379, 'm': 92.39854429, 's': 0.041887626},
{'sex': 'female', 'age': 34.5,  'l': 0.620285102, 'm': 93.02945392, 's': 0.041928568},
{'sex': 'female', 'age': 35.5,  'l': 0.57955631, 'm': 93.63382278, 's': 0.041971514},
{'sex': 'female', 'age': 36.5,  'l': 0.54198094, 'm': 94.21335709, 's': 0.042017509},
{'sex': 'female', 'age': 37.5,  'l': 0.511429832, 'm': 94.79643239, 's': 0.042104522},
{'sex': 'female', 'age': 38.5,  'l': 0.482799937, 'm': 95.37391918, 's': 0.042199507},
{'sex': 'female', 'age': 39.5,  'l': 0.455521041, 'm': 95.94692677, 's': 0.042300333},
{'sex': 'female', 'age': 40.5,  'l': 0.429150288, 'm': 96.51644912, 's': 0.042405225},
{'sex': 'female', 'age': 41.5,  'l': 0.403351725, 'm': 97.08337211, 's': 0.042512706},
{'sex': 'female', 'age': 42.5,  'l': 0.377878239, 'm': 97.6484807, 's': 0.042621565},
{'sex': 'female', 'age': 43.5,  'l': 0.352555862, 'm': 98.21246579, 's': 0.042730809},
{'sex': 'female', 'age': 44.5,  'l': 0.327270297, 'm': 98.77593069, 's': 0.042839638},
{'sex': 'female', 'age': 45.5,  'l': 0.301955463, 'm': 99.33939735, 's': 0.042947412},
{'sex': 'female', 'age': 46.5,  'l': 0.276583851, 'm': 99.9033122, 's': 0.043053626},
{'sex': 'female', 'age': 47.5,  'l': 0.251158446, 'm': 100.4680516, 's': 0.043157889},
{'sex': 'female', 'age': 48.5,  'l': 0.225705996, 'm': 101.033927, 's': 0.043259907},
{'sex': 'female', 'age': 49.5,  'l': 0.20027145, 'm': 101.6011898, 's': 0.043359463},
{'sex': 'female', 'age': 50.5,  'l': 0.174913356, 'm': 102.1700358, 's': 0.043456406},
{'sex': 'female', 'age': 51.5,  'l': 0.149700081, 'm': 102.7406094, 's': 0.043550638},
{'sex': 'female', 'age': 52.5,  'l': 0.12470671, 'm': 103.3130077, 's': 0.043642107},
{'sex': 'female', 'age': 53.5,  'l': 0.100012514, 'm': 103.8872839, 's': 0.043730791},
{'sex': 'female', 'age': 54.5,  'l': 0.075698881, 'm': 104.4634511, 's': 0.043816701},
{'sex': 'female', 'age': 55.5,  'l': 0.051847635, 'm': 105.0414853, 's': 0.043899867},
{'sex': 'female', 'age': 56.5,  'l': 0.02853967, 'm': 105.6213287, 's': 0.043980337},
{'sex': 'female', 'age': 57.5,  'l': 0.005853853, 'm': 106.2028921, 's': 0.044058171},
{'sex': 'female', 'age': 58.5,  'l': -0.016133871, 'm': 106.7860583, 's': 0.04413344},
{'sex': 'female', 'age': 59.5,  'l': -0.037351181, 'm': 107.3706841, 's': 0.044206218},
{'sex': 'female', 'age': 60.5,  'l': -0.057729947, 'm': 107.9566031, 's': 0.044276588},
{'sex': 'female', 'age': 61.5,  'l': -0.077206672, 'm': 108.5436278, 's': 0.044344632},
{'sex': 'female', 'age': 62.5,  'l': -0.09572283, 'm': 109.1315521, 's': 0.044410436},
{'sex': 'female', 'age': 63.5,  'l': -0.113225128, 'm': 109.7201531, 's': 0.044474084},
{'sex': 'female', 'age': 64.5,  'l': -0.129665689, 'm': 110.3091934, 's': 0.044535662},
{'sex': 'female', 'age': 65.5,  'l': -0.145002179, 'm': 110.8984228, 's': 0.044595254},
{'sex': 'female', 'age': 66.5,  'l': -0.159197885, 'm': 111.4875806, 's': 0.044652942},
{'sex': 'female', 'age': 67.5,  'l': -0.172221748, 'm': 112.0763967, 's': 0.044708809},
{'sex': 'female', 'age': 68.5,  'l': -0.184048358, 'm': 112.6645943, 's': 0.044762936},
{'sex': 'female', 'age': 69.5,  'l': -0.194660215, 'm': 113.2518902, 's': 0.044815402},
{'sex': 'female', 'age': 70.5,  'l': -0.204030559, 'm': 113.8380006, 's': 0.044866288},
{'sex': 'female', 'age': 71.5,  'l': -0.212174408, 'm': 114.4226317, 's': 0.044915672},
{'sex': 'female', 'age': 72.5,  'l': -0.219069129, 'm': 115.0054978, 's': 0.044963636},
{'sex': 'female', 'age': 73.5,  'l': -0.224722166, 'm': 115.5863089, 's': 0.045010259},
{'sex': 'female', 'age': 74.5,  'l': -0.229140412, 'm': 116.1647782, 's': 0.045055624},
{'sex': 'female', 'age': 75.5,  'l': -0.232335686, 'm': 116.7406221, 's': 0.045099817},
{'sex': 'female', 'age': 76.5,  'l': -0.234324563, 'm': 117.3135622, 's': 0.045142924},
{'sex': 'female', 'age': 77.5,  'l': -0.235128195, 'm': 117.8833259, 's': 0.045185036},
{'sex': 'female', 'age': 78.5,  'l': -0.234772114, 'm': 118.4496481, 's': 0.045226249},
{'sex': 'female', 'age': 79.5,  'l': -0.233286033, 'm': 119.0122722, 's': 0.045266662},
{'sex': 'female', 'age': 80.5,  'l': -0.230703633, 'm': 119.5709513, 's': 0.045306383},
{'sex': 'female', 'age': 81.5,  'l': -0.227062344, 'm': 120.1254495, 's': 0.045345524},
{'sex': 'female', 'age': 82.5,  'l': -0.222403111, 'm': 120.6755427, 's': 0.045384203},
{'sex': 'female', 'age': 83.5,  'l': -0.216770161, 'm': 121.22102, 's': 0.045422551},
{'sex': 'female', 'age': 84.5,  'l': -0.210210748, 'm': 121.7616844, 's': 0.045460702},
{'sex': 'female', 'age': 85.5,  'l': -0.202774891, 'm': 122.2973542, 's': 0.045498803},
{'sex': 'female', 'age': 86.5,  'l': -0.194515104, 'm': 122.827864, 's': 0.045537012},
{'sex': 'female', 'age': 87.5,  'l': -0.185486099, 'm': 123.3530652, 's': 0.045575495},
{'sex': 'female', 'age': 88.5,  'l': -0.175744476, 'm': 123.8728276, 's': 0.045614432},
{'sex': 'female', 'age': 89.5,  'l': -0.165348396, 'm': 124.38704, 's': 0.045654016},
{'sex': 'female', 'age': 90.5,  'l': -0.15435722, 'm': 124.8956114, 's': 0.04569445},
{'sex': 'female', 'age': 91.5,  'l': -0.142831123, 'm': 125.398472, 's': 0.045735953},
{'sex': 'female', 'age': 92.5,  'l': -0.130830669, 'm': 125.895574, 's': 0.045778759},
{'sex': 'female', 'age': 93.5,  'l': -0.118416354, 'm': 126.3868929, 's': 0.045823114},
{'sex': 'female', 'age': 94.5,  'l': -0.105648092, 'm': 126.8724284, 's': 0.04586928},
{'sex': 'female', 'age': 95.5,  'l': -0.092584657, 'm': 127.3522056, 's': 0.045917535},
{'sex': 'female', 'age': 96.5,  'l': -0.079283065, 'm': 127.8262759, 's': 0.045968169},
{'sex': 'female', 'age': 97.5,  'l': -0.065797888, 'm': 128.2947187, 's': 0.04602149},
{'sex': 'female', 'age': 98.5,  'l': -0.0521805, 'm': 128.757642, 's': 0.046077818},
{'sex': 'female', 'age': 99.5,  'l': -0.03847825, 'm': 129.2151839, 's': 0.046137487},
{'sex': 'female', 'age': 100.5,  'l': -0.024733545, 'm': 129.6675143, 's': 0.046200842},
{'sex': 'female', 'age': 101.5,  'l': -0.010982868, 'm': 130.1148354, 's': 0.04626824},
{'sex': 'female', 'age': 102.5,  'l': 0.002744306, 'm': 130.5573839, 's': 0.046340046},
{'sex': 'female', 'age': 103.5,  'l': 0.016426655, 'm': 130.995432, 's': 0.046416629},
{'sex': 'female', 'age': 104.5,  'l': 0.030052231, 'm': 131.4292887, 's': 0.046498361},
{'sex': 'female', 'age': 105.5,  'l': 0.043619747, 'm': 131.8593015, 's': 0.046585611},
{'sex': 'female', 'age': 106.5,  'l': 0.05713988, 'm': 132.2858574, 's': 0.046678741},
{'sex': 'female', 'age': 107.5,  'l': 0.070636605, 'm': 132.7093845, 's': 0.046778099},
{'sex': 'female', 'age': 108.5,  'l': 0.08414848, 'm': 133.1303527, 's': 0.04688401},
{'sex': 'female', 'age': 109.5,  'l': 0.097729873, 'm': 133.5492749, 's': 0.046996769},
{'sex': 'female', 'age': 110.5,  'l': 0.111452039, 'm': 133.9667073, 's': 0.047116633},
{'sex': 'female', 'age': 111.5,  'l': 0.125404005, 'm': 134.3832499, 's': 0.047243801},
{'sex': 'female', 'age': 112.5,  'l': 0.13969316, 'm': 134.7995463, 's': 0.047378413},
{'sex': 'female', 'age': 113.5,  'l': 0.154445482, 'm': 135.2162826, 's': 0.047520521},
{'sex': 'female', 'age': 114.5,  'l': 0.169805275, 'm': 135.634186, 's': 0.047670085},
{'sex': 'female', 'age': 115.5,  'l': 0.185934346, 'm': 136.0540223, 's': 0.047826946},
{'sex': 'female', 'age': 116.5,  'l': 0.203010488, 'm': 136.4765925, 's': 0.04799081},
{'sex': 'female', 'age': 117.5,  'l': 0.2212252, 'm': 136.9027281, 's': 0.048161228},
{'sex': 'female', 'age': 118.5,  'l': 0.240780542, 'm': 137.3332846, 's': 0.04833757},
{'sex': 'female', 'age': 119.5,  'l': 0.261885086, 'm': 137.7691339, 's': 0.048519011},
{'sex': 'female', 'age': 120.5,  'l': 0.284748919, 'm': 138.2111552, 's': 0.048704503},
{'sex': 'female', 'age': 121.5,  'l': 0.309577733, 'm': 138.6602228, 's': 0.048892759},
{'sex': 'female', 'age': 122.5,  'l': 0.336566048, 'm': 139.1171933, 's': 0.049082239},
{'sex': 'female', 'age': 123.5,  'l': 0.365889711, 'm': 139.5828898, 's': 0.049271137},
{'sex': 'female', 'age': 124.5,  'l': 0.397699038, 'm': 140.0580848, 's': 0.049457371},
{'sex': 'female', 'age': 125.5,  'l': 0.432104409, 'm': 140.5434787, 's': 0.049638596},
{'sex': 'female', 'age': 126.5,  'l': 0.46917993, 'm': 141.0396832, 's': 0.049812203},
{'sex': 'female', 'age': 127.5,  'l': 0.508943272, 'm': 141.5471945, 's': 0.049975355},
{'sex': 'female', 'age': 128.5,  'l': 0.551354277, 'm': 142.0663731, 's': 0.050125012},
{'sex': 'female', 'age': 129.5,  'l': 0.596307363, 'm': 142.59742, 's': 0.050257992},
{'sex': 'female', 'age': 130.5,  'l': 0.643626542, 'm': 143.1403553, 's': 0.050371024},
{'sex': 'female', 'age': 131.5,  'l': 0.693062173, 'm': 143.6949981, 's': 0.050460835},
{'sex': 'female', 'age': 132.5,  'l': 0.744289752, 'm': 144.2609497, 's': 0.050524236},
{'sex': 'female', 'age': 133.5,  'l': 0.79691098, 'm': 144.8375809, 's': 0.050558224},
{'sex': 'female', 'age': 134.5,  'l': 0.85045728, 'm': 145.4240246, 's': 0.050560083},
{'sex': 'female', 'age': 135.5,  'l': 0.904395871, 'm': 146.0191748, 's': 0.050527494},
{'sex': 'female', 'age': 136.5,  'l': 0.958138449, 'm': 146.621692, 's': 0.050458634},
{'sex': 'female', 'age': 137.5,  'l': 1.011054559, 'm': 147.2300177, 's': 0.050352269},
{'sex': 'female', 'age': 138.5,  'l': 1.062474568, 'm': 147.8423918, 's': 0.050207825},
{'sex': 'female', 'age': 139.5,  'l': 1.111727029, 'm': 148.4568879, 's': 0.050025434},
{'sex': 'female', 'age': 140.5,  'l': 1.158135105, 'm': 149.0714413, 's': 0.049805967},
{'sex': 'female', 'age': 141.5,  'l': 1.201050821, 'm': 149.6838943, 's': 0.049551023},
{'sex': 'female', 'age': 142.5,  'l': 1.239852328, 'm': 150.2920328, 's': 0.049262895},
{'sex': 'female', 'age': 143.5,  'l': 1.274006058, 'm': 150.8936469, 's': 0.048944504},
{'sex': 'female', 'age': 144.5,  'l': 1.303044695, 'm': 151.4865636, 's': 0.048599314},
{'sex': 'female', 'age': 145.5,  'l': 1.326605954, 'm': 152.0686985, 's': 0.048231224},
{'sex': 'female', 'age': 146.5,  'l': 1.344443447, 'm': 152.6380955, 's': 0.047844442},
{'sex': 'female', 'age': 147.5,  'l': 1.356437773, 'm': 153.1929631, 's': 0.047443362},
{'sex': 'female', 'age': 148.5,  'l': 1.362602695, 'm': 153.7317031, 's': 0.04703243},
{'sex': 'female', 'age': 149.5,  'l': 1.363085725, 'm': 154.2529332, 's': 0.046616026},
{'sex': 'female', 'age': 150.5,  'l': 1.358162799, 'm': 154.755501, 's': 0.046198356},
{'sex': 'female', 'age': 151.5,  'l': 1.348227142, 'm': 155.2384904, 's': 0.04578335},
{'sex': 'female', 'age': 152.5,  'l': 1.333772923, 'm': 155.7012216, 's': 0.045374597},
{'sex': 'female', 'age': 153.5,  'l': 1.315374704, 'm': 156.1432438, 's': 0.044975281},
{'sex': 'female', 'age': 154.5,  'l': 1.293664024, 'm': 156.564323, 's': 0.044588148},
{'sex': 'female', 'age': 155.5,  'l': 1.269304678, 'm': 156.9644258, 's': 0.044215488},
{'sex': 'female', 'age': 156.5,  'l': 1.242968236, 'm': 157.3436995, 's': 0.043859135},
{'sex': 'female', 'age': 157.5,  'l': 1.21531127, 'm': 157.7024507, 's': 0.04352048},
{'sex': 'female', 'age': 158.5,  'l': 1.186955477, 'm': 158.0411233, 's': 0.043200497},
{'sex': 'female', 'age': 159.5,  'l': 1.158471522, 'm': 158.3602756, 's': 0.042899776},
{'sex': 'female', 'age': 160.5,  'l': 1.130367088, 'm': 158.6605588, 's': 0.042618565},
{'sex': 'female', 'age': 161.5,  'l': 1.103079209, 'm': 158.9426964, 's': 0.042356812},
{'sex': 'female', 'age': 162.5,  'l': 1.076970655, 'm': 159.2074654, 's': 0.042114211},
{'sex': 'female', 'age': 163.5,  'l': 1.052329922, 'm': 159.455679, 's': 0.041890247},
{'sex': 'female', 'age': 164.5,  'l': 1.029374161, 'm': 159.688172, 's': 0.04168424},
{'sex': 'female', 'age': 165.5,  'l': 1.008254396, 'm': 159.9057871, 's': 0.041495379},
{'sex': 'female', 'age': 166.5,  'l': 0.989062282, 'm': 160.1093647, 's': 0.041322765},
{'sex': 'female', 'age': 167.5,  'l': 0.971837799, 'm': 160.299733, 's': 0.041165437},
{'sex': 'female', 'age': 168.5,  'l': 0.95657215, 'm': 160.4776996, 's': 0.041022401},
{'sex': 'female', 'age': 169.5,  'l': 0.94324228, 'm': 160.6440526, 's': 0.040892651},
{'sex': 'female', 'age': 170.5,  'l': 0.931767062, 'm': 160.7995428, 's': 0.040775193},
{'sex': 'female', 'age': 171.5,  'l': 0.922058291, 'm': 160.9448916, 's': 0.040669052},
{'sex': 'female', 'age': 172.5,  'l': 0.914012643, 'm': 161.0807857, 's': 0.040573288},
{'sex': 'female', 'age': 173.5,  'l': 0.907516917, 'm': 161.2078755, 's': 0.040487005},
{'sex': 'female', 'age': 174.5,  'l': 0.902452436, 'm': 161.3267744, 's': 0.040409354},
{'sex': 'female', 'age': 175.5,  'l': 0.898698641, 'm': 161.4380593, 's': 0.040339537},
{'sex': 'female', 'age': 176.5,  'l': 0.896143482, 'm': 161.5422726, 's': 0.040276811},
{'sex': 'female', 'age': 177.5,  'l': 0.894659668, 'm': 161.639917, 's': 0.040220488},
{'sex': 'female', 'age': 178.5,  'l': 0.89413892, 'm': 161.7314645, 's': 0.040169932},
{'sex': 'female', 'age': 179.5,  'l': 0.894475371, 'm': 161.8173534, 's': 0.040124562},
{'sex': 'female', 'age': 180.5,  'l': 0.895569834, 'm': 161.8979913, 's': 0.040083845},
{'sex': 'female', 'age': 181.5,  'l': 0.897330209, 'm': 161.9737558, 's': 0.040047295},
{'sex': 'female', 'age': 182.5,  'l': 0.899671635, 'm': 162.0449969, 's': 0.040014473},
{'sex': 'female', 'age': 183.5,  'l': 0.902516442, 'm': 162.1120386, 's': 0.03998498},
{'sex': 'female', 'age': 184.5,  'l': 0.905793969, 'm': 162.17518, 's': 0.039958458},
{'sex': 'female', 'age': 185.5,  'l': 0.909440266, 'm': 162.2346979, 's': 0.039934584},
{'sex': 'female', 'age': 186.5,  'l': 0.913397733, 'm': 162.2908474, 's': 0.039913066},
{'sex': 'female', 'age': 187.5,  'l': 0.91761471, 'm': 162.343864, 's': 0.039893644},
{'sex': 'female', 'age': 188.5,  'l': 0.922045055, 'm': 162.3939652, 's': 0.039876087},
{'sex': 'female', 'age': 189.5,  'l': 0.926647697, 'm': 162.4413513, 's': 0.039860185},
{'sex': 'female', 'age': 190.5,  'l': 0.931386217, 'm': 162.4862071, 's': 0.039845754},
{'sex': 'female', 'age': 191.5,  'l': 0.93622842, 'm': 162.5287029, 's': 0.039832629},
{'sex': 'female', 'age': 192.5,  'l': 0.941145943, 'm': 162.5689958, 's': 0.039820663},
{'sex': 'female', 'age': 193.5,  'l': 0.94611388, 'm': 162.6072309, 's': 0.039809725},
{'sex': 'female', 'age': 194.5,  'l': 0.95111043, 'm': 162.6435418, 's': 0.0397997},
{'sex': 'female', 'age': 195.5,  'l': 0.956116576, 'm': 162.6780519, 's': 0.039790485},
{'sex': 'female', 'age': 196.5,  'l': 0.961115792, 'm': 162.7108751, 's': 0.039781991},
{'sex': 'female', 'age': 197.5,  'l': 0.966093766, 'm': 162.7421168, 's': 0.039774136},
{'sex': 'female', 'age': 198.5,  'l': 0.971038162, 'm': 162.7718741, 's': 0.03976685},
{'sex': 'female', 'age': 199.5,  'l': 0.975938391, 'm': 162.8002371, 's': 0.03976007},
{'sex': 'female', 'age': 200.5,  'l': 0.980785418, 'm': 162.8272889, 's': 0.039753741},
{'sex': 'female', 'age': 201.5,  'l': 0.985571579, 'm': 162.8531067, 's': 0.039747815},
{'sex': 'female', 'age': 202.5,  'l': 0.99029042, 'm': 162.8777619, 's': 0.039742249},
{'sex': 'female', 'age': 203.5,  'l': 0.994936555, 'm': 162.9013208, 's': 0.039737004},
{'sex': 'female', 'age': 204.5,  'l': 0.999505539, 'm': 162.9238449, 's': 0.039732048},
{'sex': 'female', 'age': 205.5,  'l': 1.003993753, 'm': 162.9453912, 's': 0.039727352},
{'sex': 'female', 'age': 206.5,  'l': 1.0083983, 'm': 162.9660131, 's': 0.03972289},
{'sex': 'female', 'age': 207.5,  'l': 1.012716921, 'm': 162.9857599, 's': 0.03971864},
{'sex': 'female', 'age': 208.5,  'l': 1.016947912, 'm': 163.0046776, 's': 0.039714581},
{'sex': 'female', 'age': 209.5,  'l': 1.021090055, 'm': 163.0228094, 's': 0.039710697},
{'sex': 'female', 'age': 210.5,  'l': 1.025142554, 'm': 163.0401953, 's': 0.039706971},
{'sex': 'female', 'age': 211.5,  'l': 1.029104983, 'm': 163.0568727, 's': 0.039703391},
{'sex': 'female', 'age': 212.5,  'l': 1.032977233, 'm': 163.0728768, 's': 0.039699945},
{'sex': 'female', 'age': 213.5,  'l': 1.036759475, 'm': 163.0882404, 's': 0.039696623},
{'sex': 'female', 'age': 214.5,  'l': 1.040452117, 'm': 163.1029943, 's': 0.039693415},
{'sex': 'female', 'age': 215.5,  'l': 1.044055774, 'm': 163.1171673, 's': 0.039690313},
{'sex': 'female', 'age': 216.5,  'l': 1.047571238, 'm': 163.1307866, 's': 0.039687311},
{'sex': 'female', 'age': 217.5,  'l': 1.050999451, 'm': 163.1438776, 's': 0.039684402},
{'sex': 'female', 'age': 218.5,  'l': 1.054341482, 'm': 163.1564644, 's': 0.039681581},
{'sex': 'female', 'age': 219.5,  'l': 1.057598512, 'm': 163.1685697, 's': 0.039678842},
{'sex': 'female', 'age': 220.5,  'l': 1.060771808, 'm': 163.1802146, 's': 0.039676182},
{'sex': 'female', 'age': 221.5,  'l': 1.063862715, 'm': 163.1914194, 's': 0.039673596},
{'sex': 'female', 'age': 222.5,  'l': 1.066872639, 'm': 163.202203, 's': 0.039671082},
{'sex': 'female', 'age': 223.5,  'l': 1.069803036, 'm': 163.2125835, 's': 0.039668635},
{'sex': 'female', 'age': 224.5,  'l': 1.072655401, 'm': 163.2225779, 's': 0.039666254},
{'sex': 'female', 'age': 225.5,  'l': 1.075431258, 'm': 163.2322024, 's': 0.039663936},
{'sex': 'female', 'age': 226.5,  'l': 1.078132156, 'm': 163.2414722, 's': 0.039661679},
{'sex': 'female', 'age': 227.5,  'l': 1.080759655, 'm': 163.2504019, 's': 0.039659481},
{'sex': 'female', 'age': 228.5,  'l': 1.083315329, 'm': 163.2590052, 's': 0.039657339},
{'sex': 'female', 'age': 229.5,  'l': 1.085800751, 'm': 163.2672954, 's': 0.039655252},
{'sex': 'female', 'age': 230.5,  'l': 1.088217496, 'm': 163.2752848, 's': 0.039653218},
{'sex': 'female', 'age': 231.5,  'l': 1.090567133, 'm': 163.2829854, 's': 0.039651237},
{'sex': 'female', 'age': 232.5,  'l': 1.092851222, 'm': 163.2904086, 's': 0.039649306},
{'sex': 'female', 'age': 233.5,  'l': 1.095071313, 'm': 163.297565, 's': 0.039647424},
{'sex': 'female', 'age': 234.5,  'l': 1.097228939, 'm': 163.304465, 's': 0.039645591},
{'sex': 'female', 'age': 235.5,  'l': 1.099325619, 'm': 163.3111185, 's': 0.039643804},
{'sex': 'female', 'age': 236.5,  'l': 1.101362852, 'm': 163.3175349, 's': 0.039642063},
{'sex': 'female', 'age': 237.5,  'l': 1.103342119, 'm': 163.3237231, 's': 0.039640367},
{'sex': 'female', 'age': 238.5,  'l': 1.105264876, 'm': 163.3296918, 's': 0.039638715},
{'sex': 'female', 'age': 239.5,  'l': 1.107132561, 'm': 163.3354491, 's': 0.039637105},
{'sex': 'female', 'age': 240,  'l': 1.108046193, 'm': 163.338251, 's': 0.039636316}
]};


