/*******************************************************************************
 * Copyright (c) 2011 Coverity, Inc

 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Coverity, Inc - initial implementation and documentation
 *******************************************************************************/
package jenkins.plugins.coverity;

import com.coverity.ws.v5.MergedDefectDataObj;
import hudson.Util;
import hudson.model.Descriptor;
import org.kohsuke.stapler.DataBoundConstructor;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * Responsible for filtering the full list of defects to determine if a build should fail or not.
 * Filters are inclusive: a defect needs to pass every filter in order to be included.
 */
public class DefectFilters {
	private List<String> classifications;
	private List<String> actions;
	private List<String> severities;
	private List<String> components;
	private List<String> checkers;
	private List<String> ignoredCheckers;
	private Date cutOffDate;

	@DataBoundConstructor
	public DefectFilters(List<String> actions, List<String> classifications, List<String> severities, List<String> components, List<String> checkers, String cutOffDate) throws Descriptor.FormException {
		this.classifications = Util.fixNull(classifications);
		this.actions = Util.fixNull(actions);
		this.severities = Util.fixNull(severities);
		this.components = Util.fixNull(components);
		this.checkers = Util.fixNull(checkers);
		this.ignoredCheckers = new ArrayList<String>();

		cutOffDate = Util.fixEmpty(cutOffDate);
		if(cutOffDate != null) {
			try {
				this.cutOffDate = new SimpleDateFormat("yyyy-MM-dd").parse(cutOffDate);
			} catch(ParseException e) {
				this.cutOffDate = null;
			}
		} else {
			this.cutOffDate = null;
		}
	}

	void invertCheckers(Set<String> allCheckers, List<String> allClassifications, List<String> allActions, List<String> allSeverities, List<String> allComponents) {
		if(classifications.isEmpty() && checkers.isEmpty() && actions.isEmpty() && components.isEmpty() && severities.isEmpty()) {
			ignoredCheckers = new ArrayList<String>();
			actions = allActions;
			severities = allSeverities;
			components = allComponents;
			classifications = allClassifications;
		} else {
			ignoredCheckers = new ArrayList<String>(allCheckers);
			ignoredCheckers.removeAll(checkers);
			checkers = null;
		}
	}

	public boolean isClassificationSelected(String action) {
		return classifications.contains(action);
	}

	public boolean isActionSelected(String action) {
		return actions.contains(action);
	}

	public boolean isSeveritySelected(String severity) {
		return severities.contains(severity);
	}

	public boolean isComponentSelected(String component) {
		return components.contains(component);
	}

	public boolean isCheckerSelected(String checker) {
		if(ignoredCheckers == null) {
			return false;
		}
		return !ignoredCheckers.contains(checker);
	}

	public String getCutOffDate() {
		if(cutOffDate == null) return null;
		return new SimpleDateFormat("yyyy-MM-dd").format(cutOffDate);
	}

	public boolean matches(MergedDefectDataObj defect) {
		return isActionSelected(defect.getAction()) &&
				isClassificationSelected(defect.getClassification()) &&
				isSeveritySelected(defect.getSeverity()) &&
				isComponentSelected(defect.getComponentName()) &&
				isCheckerSelected(defect.getCheckerName()) &&
				Arrays.asList("New", "Triaged", "Various").contains(defect.getStatus()) &&
				(cutOffDate == null || defect.getFirstDetected().toGregorianCalendar().getTime().after(cutOffDate));
	}

	@Override
	public boolean equals(Object o) {
		if(this == o) return true;
		if(o == null || getClass() != o.getClass()) return false;

		DefectFilters that = (DefectFilters) o;

		if(actions != null ? !actions.equals(that.actions) : that.actions != null) return false;
		if(checkers != null ? !checkers.equals(that.checkers) : that.checkers != null) return false;
		if(classifications != null ? !classifications.equals(that.classifications) : that.classifications != null)
			return false;
		if(components != null ? !components.equals(that.components) : that.components != null) return false;
		if(cutOffDate != null ? !cutOffDate.equals(that.cutOffDate) : that.cutOffDate != null) return false;
		if(severities != null ? !severities.equals(that.severities) : that.severities != null) return false;

		return true;
	}

	@Override
	public int hashCode() {
		int result = classifications != null ? classifications.hashCode() : 0;
		result = 31 * result + (actions != null ? actions.hashCode() : 0);
		result = 31 * result + (severities != null ? severities.hashCode() : 0);
		result = 31 * result + (components != null ? components.hashCode() : 0);
		result = 31 * result + (checkers != null ? checkers.hashCode() : 0);
		result = 31 * result + (cutOffDate != null ? cutOffDate.hashCode() : 0);
		return result;
	}
}
