/**
 *
 * Copyright the original author or authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jivesoftware.smack.provider;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jivesoftware.smack.packet.ExtensionElement;
import org.jivesoftware.smack.packet.IQ;
import org.jivesoftware.smack.util.PacketParserUtils;
import org.jivesoftware.smack.xml.XmlPullParser;

/**
 * Loads the {@link IqProvider} and {@link ExtensionElementProvider} information from a standard provider file in preparation
 * for loading into the {@link ProviderManager}.
 *
 * @author Robin Collier
 *
 */
public class ProviderFileLoader implements ProviderLoader {
    private static final Logger LOGGER = Logger.getLogger(ProviderFileLoader.class.getName());

    private final Collection<IQProviderInfo> iqProviders = new ArrayList<IQProviderInfo>();
    private final Collection<ExtensionProviderInfo> extProviders  = new ArrayList<ExtensionProviderInfo>();
    private final Collection<StreamFeatureProviderInfo> sfProviders = new ArrayList<StreamFeatureProviderInfo>();

    private List<Exception> exceptions = new ArrayList<Exception>();

    public ProviderFileLoader(InputStream providerStream) {
        this(providerStream, ProviderFileLoader.class.getClassLoader());
    }

    @SuppressWarnings("unchecked")
    public ProviderFileLoader(InputStream providerStream, ClassLoader classLoader) {
        // Load processing providers.
        try (InputStream is = providerStream) {
            XmlPullParser parser = PacketParserUtils.getParserFor(is);
            XmlPullParser.Event eventType = parser.getEventType();
            do {
                if (eventType == XmlPullParser.Event.START_ELEMENT) {
                    final String typeName = parser.getName();

                    try {
                        if (!"smackProviders".equals(typeName)) {
                            parser.next();
                            parser.next();
                            String elementName = parser.nextText();
                            parser.next();
                            parser.next();
                            String namespace = parser.nextText();
                            parser.next();
                            parser.next();
                            String className = parser.nextText();

                            try {
                                final Class<?> provider = classLoader.loadClass(className);
                                switch (typeName) {
                                case "iqProvider":
                                    // Attempt to load the provider class and then create
                                    // a new instance if it's an IqProvider. Otherwise, if it's
                                    // an IQ class, add the class object itself, then we'll use
                                    // reflection later to create instances of the class.
                                    // Add the provider to the map.
                                    if (IqProvider.class.isAssignableFrom(provider)) {
                                        IqProvider<IQ> iqProvider = (IqProvider<IQ>) provider.getConstructor().newInstance();
                                        iqProviders.add(new IQProviderInfo(elementName, namespace, iqProvider));
                                    }
                                    else {
                                        exceptions.add(new IllegalArgumentException(className + " is not a IqProvider"));
                                    }
                                    break;
                                case "extensionProvider":
                                    // Attempt to load the provider class and then create
                                    // a new instance if it's an ExtensionProvider. Otherwise, if it's
                                    // a PacketExtension, add the class object itself and
                                    // then we'll use reflection later to create instances
                                    // of the class.
                                    if (ExtensionElementProvider.class.isAssignableFrom(provider)) {
                                        ExtensionElementProvider<ExtensionElement> extensionElementProvider = (ExtensionElementProvider<ExtensionElement>) provider.getConstructor().newInstance();
                                        extProviders.add(new ExtensionProviderInfo(elementName, namespace,
                                                        extensionElementProvider));
                                    }
                                    else {
                                        exceptions.add(new IllegalArgumentException(className
                                                        + " is not a PacketExtensionProvider"));
                                    }
                                    break;
                                case "streamFeatureProvider":
                                    ExtensionElementProvider<ExtensionElement> streamFeatureProvider = (ExtensionElementProvider<ExtensionElement>) provider.getConstructor().newInstance();
                                    sfProviders.add(new StreamFeatureProviderInfo(elementName,
                                                    namespace,
                                                    streamFeatureProvider));
                                    break;
                                default:
                                    LOGGER.warning("Unknown provider type: " + typeName);
                                }
                            }
                            catch (ClassNotFoundException cnfe) {
                                LOGGER.log(Level.SEVERE, "Could not find provider class", cnfe);
                                exceptions.add(cnfe);
                            }
                            catch (InstantiationException ie) {
                                LOGGER.log(Level.SEVERE, "Could not instanciate " + className, ie);
                                exceptions.add(ie);
                            }
                        }
                    }
                    catch (IllegalArgumentException illExc) {
                        LOGGER.log(Level.SEVERE, "Invalid provider type found [" + typeName + "] when expecting iqProvider or extensionProvider", illExc);
                        exceptions.add(illExc);
                    }
                }
                eventType = parser.next();
            }
            while (eventType != XmlPullParser.Event.END_DOCUMENT);
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Unknown error occurred while parsing provider file", e);
            exceptions.add(e);
        }
    }

    @Override
    public Collection<IQProviderInfo> getIQProviderInfo() {
        return iqProviders;
    }

    @Override
    public Collection<ExtensionProviderInfo> getExtensionProviderInfo() {
        return extProviders;
    }

    @Override
    public Collection<StreamFeatureProviderInfo> getStreamFeatureProviderInfo() {
        return sfProviders;
    }

    public List<Exception> getLoadingExceptions() {
        return Collections.unmodifiableList(exceptions);
    }
}
