/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection;

import static org.iherus.shiro.cache.redis.Constant.DEFAULT_DEL_BATCH_SIZE;
import static org.iherus.shiro.cache.redis.Constant.DEFAULT_FETCH_BATCH_SIZE;
import static org.iherus.shiro.cache.redis.Constant.DEFAULT_SCAN_BATCH_SIZE;

/**
 * 批量处理参数类
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class BatchOptions {

	public static final BatchOptions defaulted = new BatchOptions(DEFAULT_SCAN_BATCH_SIZE, DEFAULT_DEL_BATCH_SIZE,
			DEFAULT_FETCH_BATCH_SIZE);

	private final int scanBatchSize;
	private final int deleteBatchSize;
	private final int fetchBatchSize;

	BatchOptions(Builder builder) {
		this(builder.scanBatchSize, builder.deleteBatchSize, builder.fetchBatchSize);
	}

	private BatchOptions(int scanBatchSize, int deleteBatchSize, int fetchBatchSize) {
		this.scanBatchSize = scanBatchSize;
		this.deleteBatchSize = deleteBatchSize;
		this.fetchBatchSize = fetchBatchSize;
	}

	public int getScanBatchSize() {
		return scanBatchSize;
	}

	public int getDeleteBatchSize() {
		return deleteBatchSize;
	}

	public int getFetchBatchSize() {
		return fetchBatchSize;
	}

	public static Builder builder() {
		return new Builder();
	}

	public static class Builder {

		private int scanBatchSize;
		private int deleteBatchSize;
		private int fetchBatchSize;

		private Builder() {
			this.scanBatchSize = DEFAULT_SCAN_BATCH_SIZE;
			this.deleteBatchSize = DEFAULT_DEL_BATCH_SIZE;
			this.fetchBatchSize = DEFAULT_FETCH_BATCH_SIZE;
		}

		public Builder scanSize(int size) {
			this.scanBatchSize = Math.max(1, size);
			return this;
		}

		public Builder deleteSize(int size) {
			this.deleteBatchSize = Math.max(1, size);
			return this;
		}

		public Builder fetchSize(int size) {
			this.fetchBatchSize = Math.max(1, size);
			return this;
		}

		public BatchOptions build() {
			return new BatchOptions(this);
		}

	}

}
