/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.jedis;

import static org.iherus.shiro.cache.redis.Constant.GETDEL;
import static org.iherus.shiro.cache.redis.Constant.GETSET;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.util.RedisVerUtils;
import org.iherus.shiro.util.Utils;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.util.SafeEncoder;

/**
 * JedisConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class JedisConnection extends AbstractJedisConnection implements RedisConnection {

	private final Jedis jedis;
	private final BatchOptions options;

	private Optional<Integer> database = Optional.empty();

	public JedisConnection(Jedis jedis) {
		this(jedis, BatchOptions.defaulted);
	}

	public JedisConnection(Jedis jedis, BatchOptions options) {
		this.jedis = jedis;
		this.options = options;
	}

	public Jedis getJedis() {
		return jedis;
	}

	public Optional<Integer> getDatabase() {
		return database;
	}

	public void setDatabase(Integer database) {
		this.database = Optional.ofNullable(database == null ? null : Math.max(0, database));
	}

	public BatchOptions getOptions() {
		return options;
	}

	protected Jedis getCommandExecutor() {
		getDatabase().filter(db -> db.intValue() != jedis.getDB()).ifPresent(jedis::select);
		return jedis;
	}

	@Override
	public byte[] get(byte[] key) {
		return getCommandExecutor().get(key);
	}

	@Override
	public byte[] set(byte[] key, byte[] value, Duration expired) {
		List<byte[]> args = Arrays.asList(value, Utils.longToBytes(expired.toMillis()));
		return (byte[]) getCommandExecutor().eval(SafeEncoder.encode(GETSET), Collections.singletonList(key), args);
	}

	@Override
	public Long mdel(byte[]... keys) {
		if (Utils.isEmpty(keys)) return 0L;

		boolean unlink = RedisVerUtils.getServerVersion(() -> {
			return parseServerVersion(getJedis().info("Server"));
		}).isSupportUnlink();

		return batchDeleteOnStandalone(this.options.getDeleteBatchSize(), keys, (batchKeys) -> {
			return unlink ? getCommandExecutor().unlink(batchKeys) : getCommandExecutor().del(batchKeys);
		});
	}

	@Override
	public List<byte[]> mget(byte[]... keys) {
		return batchGetOnStandalone(this.options.getFetchBatchSize(), keys, (batchKeys) -> {
			return getCommandExecutor().mget(batchKeys);
		});
	}

	@Override
	public byte[] del(byte[] key) {
		return (byte[]) getCommandExecutor().eval(SafeEncoder.encode(GETDEL), 1, key);
	}

	@Override
	public Set<byte[]> keys(byte[] pattern) {
		return scanKeys(getCommandExecutor(), pattern, this.options.getScanBatchSize());
	}

	@Override
	public void close() {
		releaseConnection(getJedis());
	}

}
