/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.redisson;

import static org.iherus.shiro.util.Utils.bytesToText;
import static org.iherus.shiro.util.Utils.intToBytes;
import static org.iherus.shiro.util.Utils.newMutableArray;
import static org.redisson.client.codec.ByteArrayCodec.INSTANCE;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.iherus.shiro.cache.redis.Constant;
import org.iherus.shiro.cache.redis.connection.AbstractRedisConnection;
import org.iherus.shiro.util.Utils.MutableArray;
import org.redisson.BaseIterator;
import org.redisson.ScanResult;
import org.redisson.api.RFuture;
import org.redisson.client.RedisClient;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.client.protocol.decoder.ListScanResult;
import org.redisson.command.CommandExecutor;
import org.redisson.connection.MasterSlaveEntry;

/**
 * AbstractRedissonConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public abstract class AbstractRedissonConnection extends AbstractRedisConnection {

	protected abstract CommandExecutor getCommandExecutor();

	public Iterator<byte[]> scanKeys(MasterSlaveEntry entry, final byte[] pattern, final int batchSize) {

		return new BaseIterator<byte[], byte[]>() {

			@Override
			protected ScanResult<byte[]> iterator(RedisClient client, long nextIterPos) {
				return getCommandExecutor().get(scanAsync(client, entry, nextIterPos, pattern, batchSize));
			}

			@Override
			protected byte[] getValue(byte[] entry) {
				return entry;
			}

			@Override
			protected void remove(byte[] value) {
				// The element has been removed internally and no processing is done here.
			}

		};

	}

	private RFuture<ListScanResult<byte[]>> scanAsync(RedisClient client, MasterSlaveEntry entry, long startPos,
			byte[] pattern, int count) {
		return getCommandExecutor().readAsync(client, entry, INSTANCE, RedisCommands.SCAN, startPos, "MATCH",
				((Object) pattern), "COUNT", count);
	}
	
	@SuppressWarnings("unchecked")
	public Set<byte[]> scanKeysOnDb(MasterSlaveEntry entry, final int db, final byte[] pattern,
			final int batchSize) {
		Set<byte[]> keys = new HashSet<byte[]>();

		final MutableArray<Object> mutableArray = newMutableArray(null, (Object) pattern,
				(Object) intToBytes(batchSize), (Object) intToBytes(db));

		byte[] cursor = intToBytes(0);

		do {
			Object[] params = mutableArray.replace(0, (Object) cursor).toArray();

			RFuture<Object> f = getCommandExecutor().evalReadAsync(entry, INSTANCE, RedisCommands.EVAL_LIST,
					Constant.Select.SCAN.command(), Collections.emptyList(), params);

			List<Object> objectList = (List<Object>) getCommandExecutor().get(f);

			cursor = (byte[]) objectList.get(0);
			keys.addAll((Collection<? extends byte[]>) objectList.get(1));

		} while (bytesToText(cursor).compareTo("0") > 0);

		return keys;
	}

}
