/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.redisson;

import static org.iherus.shiro.util.Utils.assertNotNull;
import static org.iherus.shiro.util.Utils.invokeMethod;
import static org.iherus.shiro.util.Utils.isNotEmpty;

import java.net.URL;
import java.time.Duration;
import java.util.Optional;
import java.util.Set;

import org.apache.shiro.util.ClassUtils;
import org.iherus.shiro.cache.redis.config.RedisClusterConfiguration;
import org.iherus.shiro.cache.redis.config.RedisConfiguration;
import org.iherus.shiro.cache.redis.config.RedisSentinelConfiguration;
import org.iherus.shiro.cache.redis.config.RedisStandaloneConfiguration;
import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.Destroyable;
import org.iherus.shiro.cache.redis.connection.Initializable;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.cache.redis.connection.RedisConnectionFactory;
import org.iherus.shiro.exception.InvocationException;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.ByteArrayCodec;
import org.redisson.config.BaseMasterSlaveServersConfig;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.Config;
import org.redisson.config.SentinelServersConfig;
import org.redisson.config.SingleServerConfig;

/**
 * Implementation of {@link RedisConnectionFactory} a based on {@literal Redisson}.
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class RedissonConnectionFactory implements RedisConnectionFactory, Initializable, Destroyable {

	private volatile RedissonClient client;

	private Optional<String> clientName = Optional.empty();

	/**
	 * Minimum idle Redis connection amount
	 */
	private Optional<Integer> connectionMinIdleSize = Optional.empty();

	/**
	 * Redis connection maximum pool size
	 */
	private Optional<Integer> connectionPoolSize = Optional.empty();

	private Optional<Duration> soTimeout = Optional.empty();
	private Optional<Duration> connectTimeout = Optional.empty();

	/**
	 * SSL Parameters
	 */
	private boolean useSsl;
	private URL sslTruststore;
	private String sslTruststorePassword;
	private URL sslKeystore;
	private String sslKeystorePassword;

	private RedisConfiguration configuration = RedisConfiguration.defaulted;

	private Optional<BatchOptions> batchOptions = Optional.empty();

	private final Object lock = new Object();

	public RedissonConnectionFactory() {

	}

	public RedissonConnectionFactory(RedisStandaloneConfiguration standaloneConfig) {
		assertNotNull(standaloneConfig, "RedisStandaloneConfiguration must not be null.");
		this.configuration = standaloneConfig;
	}

	public RedissonConnectionFactory(RedisSentinelConfiguration sentinelConfig) {
		assertNotNull(sentinelConfig, "RedisSentinelConfiguration must not be null.");
		this.configuration = sentinelConfig;
	}

	public RedissonConnectionFactory(RedisClusterConfiguration clusterConfig) {
		assertNotNull(clusterConfig, "RedisClusterConfiguration must not be null.");
		this.configuration = clusterConfig;
	}

	public Optional<Integer> getConnectionMinIdleSize() {
		return connectionMinIdleSize;
	}

	public void setConnectionMinIdleSize(Integer connectionMinIdleSize) {
		this.connectionMinIdleSize = Optional.of(Math.max(0, connectionMinIdleSize));
	}

	public Optional<Integer> getConnectionPoolSize() {
		return connectionPoolSize;
	}

	public void setConnectionPoolSize(Integer connectionPoolSize) {
		this.connectionPoolSize = Optional.of(Math.max(8, connectionPoolSize));
	}

	public Optional<String> getClientName() {
		return clientName;
	}

	public void setClientName(String clientName) {
		this.clientName = Optional.ofNullable(clientName);
	}

	public RedisConfiguration getConfiguration() {
		return configuration;
	}

	protected RedisConfiguration getRequiredConfig() {
		RedisConfiguration config = getConfiguration();
		if (config == null) {
			throw new IllegalArgumentException("RedisConfiguration must not be null.");
		}
		return config;
	}

	public void setConfiguration(RedisConfiguration configuration) {
		this.configuration = configuration;
	}

	public Optional<Duration> getSoTimeout() {
		return soTimeout;
	}

	public void setSoTimeout(Duration soTimeout) {
		this.soTimeout = Optional.ofNullable(soTimeout);
	}

	public void setSoTimeoutMillis(long millis) {
		this.setSoTimeout(Duration.ofMillis(millis));
	}

	public Optional<Duration> getConnectTimeout() {
		return connectTimeout;
	}

	public void setConnectTimeout(Duration connectTimeout) {
		this.connectTimeout = Optional.ofNullable(connectTimeout);
	}

	public void setConnectTimeoutMillis(long millis) {
		this.setConnectTimeout(Duration.ofMillis(millis));
	}

	public URL getSslTruststore() {
		return sslTruststore;
	}

	public void setSslTruststore(URL sslTruststore) {
		this.sslTruststore = sslTruststore;
	}

	public String getSslTruststorePassword() {
		return sslTruststorePassword;
	}

	public void setSslTruststorePassword(String sslTruststorePassword) {
		this.sslTruststorePassword = sslTruststorePassword;
	}

	public URL getSslKeystore() {
		return sslKeystore;
	}

	public void setSslKeystore(URL sslKeystore) {
		this.sslKeystore = sslKeystore;
	}

	public String getSslKeystorePassword() {
		return sslKeystorePassword;
	}

	public void setSslKeystorePassword(String sslKeystorePassword) {
		this.sslKeystorePassword = sslKeystorePassword;
	}

	public Optional<BatchOptions> getBatchOptions() {
		return batchOptions;
	}

	@Override
	public void setBatchOptions(BatchOptions options) {
		this.batchOptions = Optional.ofNullable(options);
	}
	
	public boolean isUseSsl() {
		return useSsl;
	}

	public void setUseSsl(boolean useSsl) {
		this.useSsl = useSsl;
	}

	@Override
	public void init() throws Exception {
		if (this.client == null) {
			this.client = createClient();
		}
	}

	public RedissonClient getClient() {
		if (client == null) {
			synchronized (lock) {
				if (client == null) {
					this.client = createClient();
				}
			}
		}
		return client;
	}

	@Override
	public void destroy() throws Exception {
		try {
			if (this.client != null) this.client.shutdown();
		} finally {
			this.client = null;
		}
	}

	@Override
	public RedisConnection getConnection() {
		if (isRedisClusterAware()) {
			return getClusterConnection();
		}
		return new RedissonConnection(getClient(), this.batchOptions.orElse(BatchOptions.defaulted));
	}

	public RedissonClusterConnection getClusterConnection() {
		if (!isRedisClusterAware()) {
			throw new InvocationException("Cluster is not configured.");
		}
		return new RedissonClusterConnection(getClient(), this.batchOptions.orElse(BatchOptions.defaulted));
	}

	public boolean isRedisSentinelAware() {
		return RedisConfiguration.isSentinelConfiguration(configuration);
	}

	public boolean isRedisClusterAware() {
		return RedisConfiguration.isClusterConfiguration(configuration);
	}

	protected RedissonClient createClient() {

		Config config = new Config();

		if (isRedisSentinelAware()) {

			RedisSentinelConfiguration settingConfig = (RedisSentinelConfiguration) getRequiredConfig();

			SentinelServersConfig sentinelConfig = config.useSentinelServers();

			Set<String> sentinels = settingConfig.getAddresses(isUseSsl());
			sentinelConfig.setMasterName(settingConfig.getMasterName());
			sentinelConfig.setDatabase(settingConfig.getDatabase());
			sentinelConfig.addSentinelAddress(sentinels.toArray(new String[sentinels.size()]));
			setCommonProperties(sentinelConfig, settingConfig.getPassword());

		} else if (isRedisClusterAware()) {

			RedisClusterConfiguration settingConfig = (RedisClusterConfiguration) getRequiredConfig();

			ClusterServersConfig clusterConfig = config.useClusterServers();

			Set<String> nodes = settingConfig.getAddresses(isUseSsl());
			clusterConfig.addNodeAddress(nodes.toArray(new String[nodes.size()]));
			clusterConfig.setRetryAttempts(settingConfig.getMaxAttempts());

			setCommonProperties(clusterConfig, settingConfig.getPassword());

		} else {

			RedisStandaloneConfiguration settingConfig = (RedisStandaloneConfiguration) getRequiredConfig();

			SingleServerConfig standaloneConfig = config.useSingleServer();

			standaloneConfig.setAddress(settingConfig.getAddress(isUseSsl()));

			standaloneConfig.setDatabase(settingConfig.getDatabase());

			setCommonProperties(standaloneConfig, settingConfig.getPassword());
		}
		
		return Redisson.create(config.setCodec(ByteArrayCodec.INSTANCE));
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	protected void setCommonProperties(Object config, String password) {

		Class clazz = ClassUtils.forName("org.redisson.config.BaseConfig");

		Class configClass = config.getClass();

		if (BaseMasterSlaveServersConfig.class.isAssignableFrom(configClass)) {
			BaseMasterSlaveServersConfig configToUse = (BaseMasterSlaveServersConfig) config;

			getConnectionMinIdleSize().ifPresent(configToUse::setMasterConnectionMinimumIdleSize);
			getConnectionMinIdleSize().ifPresent(configToUse::setSlaveConnectionMinimumIdleSize);

			getConnectionPoolSize().ifPresent(configToUse::setMasterConnectionPoolSize);
			getConnectionPoolSize().ifPresent(configToUse::setSlaveConnectionPoolSize);

		} else if (SingleServerConfig.class.isAssignableFrom(configClass)) {
			SingleServerConfig configToUse = (SingleServerConfig) config;

			getConnectionMinIdleSize().ifPresent(configToUse::setConnectionMinimumIdleSize);
			getConnectionPoolSize().ifPresent(configToUse::setConnectionPoolSize);

		}

		if (clazz.isAssignableFrom(config.getClass())) {
			getClientName().ifPresent(name -> invokeMethod(config, "setClientName", String.class, name));
			getConnectTimeout().ifPresent(timeout -> invokeMethod(config, "setConnectTimeout", Integer.class, timeout));
			getSoTimeout().ifPresent(timeout -> invokeMethod(config, "setTimeout", Integer.class, timeout));

			Optional.ofNullable(password).ifPresent(pwd -> invokeMethod(config, "setPassword", String.class, pwd));
			Optional.ofNullable(getSslTruststore())
					.ifPresent(it -> invokeMethod(config, "setSslTruststore", URL.class, it));
			Optional.ofNullable(getSslKeystore())
					.ifPresent(it -> invokeMethod(config, "setSslKeystore", URL.class, it));

			if (isNotEmpty(getSslTruststorePassword())) {
				invokeMethod(config, "setSslTruststorePassword", String.class, getSslTruststorePassword());
			}

			if (isNotEmpty(getSslKeystorePassword())) {
				invokeMethod(config, "setSslKeystorePassword", String.class, getSslKeystorePassword());
			}
		}
	}

}
