/*
 * Decompiled with CFR 0.152.
 */
package org.iherus.shiro.cache.redis;

import java.io.Serializable;
import java.lang.ref.SoftReference;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Iterator;
import java.util.Optional;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.shiro.session.Session;
import org.iherus.shiro.util.concurrent.ConcurrentLinkedHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MemorySessionCache {
    private static final Logger logger = LoggerFactory.getLogger(MemorySessionCache.class);
    private static final int DEFAULT_INITIAL_CAPACITY = 1000;
    private static final long DEFAULT_MAX_CAPACITY = 131072L;
    private int ttl = 1000;
    private long capacityThreshold = 131072L;
    private ConcurrentMap<Serializable, SoftReference<LocalSession>> cache;
    private AtomicBoolean watchdogEnabled = new AtomicBoolean(false);
    private volatile ScheduledExecutorService executorService;

    MemorySessionCache() {
    }

    public long getCapacityThreshold() {
        return this.capacityThreshold;
    }

    public void setCapacityThreshold(long capacityThreshold) {
        this.capacityThreshold = capacityThreshold;
    }

    public int getTtl() {
        return this.ttl;
    }

    public void setTtl(int ttl) {
        this.ttl = ttl;
    }

    public void init() {
        ConcurrentLinkedHashMap.Builder builder = new ConcurrentLinkedHashMap.Builder();
        builder.initialCapacity(1000);
        Optional.of(this.capacityThreshold).filter(c -> c >= 0L).ifPresent(builder::maximumWeightedCapacity);
        this.cache = builder.build();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ScheduledExecutorService getExecutorService() {
        if (this.executorService == null) {
            MemorySessionCache memorySessionCache = this;
            synchronized (memorySessionCache) {
                if (this.executorService == null) {
                    this.executorService = Executors.newSingleThreadScheduledExecutor();
                }
            }
        }
        return this.executorService;
    }

    public Session get(Serializable key) {
        SoftReference ref;
        if (logger.isDebugEnabled()) {
            logger.debug("Getting a session instance with key [{}] from the local cache.", (Object)key);
        }
        if ((ref = (SoftReference)this.cache.get(key)) != null) {
            LocalSession localSession = (LocalSession)ref.get();
            if (localSession.isValid()) {
                return localSession.getSession();
            }
            if (!this.watchdogEnabled.get()) {
                this.cache.remove(key);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Session put(Serializable key, Session session) {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Putting a session instance with key [{}] to the local cache.", (Object)key);
            }
            Session session2 = MemorySessionCache.unwrap(this.cache.put(key, MemorySessionCache.wrap(session, this.getTtl())));
            return session2;
        }
        finally {
            if (this.watchdogEnabled.compareAndSet(false, true)) {
                this.getExecutorService().schedule(() -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Executing a cleanup job for a locally cached session.");
                    }
                    try {
                        Iterator iterator = this.cache.entrySet().iterator();
                        while (iterator.hasNext()) {
                            LocalSession localSession = (LocalSession)((SoftReference)iterator.next().getValue()).get();
                            if (localSession.isValid()) continue;
                            iterator.remove();
                        }
                    }
                    finally {
                        this.watchdogEnabled.compareAndSet(true, false);
                    }
                }, 3000L, TimeUnit.MILLISECONDS);
            }
        }
    }

    public Session remove(Serializable key) {
        return MemorySessionCache.unwrap((SoftReference)this.cache.remove(key));
    }

    public void clear() {
        this.cache.clear();
    }

    public int size() {
        return this.cache.size();
    }

    private static SoftReference<LocalSession> wrap(Session session, long ttl) {
        LocalDateTime expiredTime = LocalDateTime.now().plus(ttl, ChronoUnit.MILLIS);
        return new SoftReference<LocalSession>(new LocalSession(session, expiredTime));
    }

    private static Session unwrap(SoftReference<LocalSession> ref) {
        return ref == null ? null : ref.get().getSession();
    }

    private static class LocalSession {
        final Session session;
        final LocalDateTime expiredTime;

        LocalSession(Session session, LocalDateTime expiredTime) {
            this.session = session;
            this.expiredTime = expiredTime;
        }

        public Session getSession() {
            return this.session;
        }

        boolean isValid() {
            return LocalDateTime.now().isBefore(this.expiredTime);
        }
    }
}

