/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.config;

import java.io.Serializable;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

import org.iherus.shiro.util.Utils;

public class HostPortPair implements Serializable {

	private static final long serialVersionUID = 8603816125320062450L;

	private static final String PROTOCOL_REDIS = "redis://";
	private static final String PROTOCOL_REDISS = "rediss://";

	private String host;
	private int port;

	protected HostPortPair() {

	}

	public HostPortPair(String host, int port) {
		this.host = host;
		this.port = port;
	}

	public String getHost() {
		return host;
	}

	public int getPort() {
		return port;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + Utils.nullSafeHashCode(host);
		result = prime * result + Utils.nullSafeHashCode(port);
		return result;
	}

	@Override
	public boolean equals(Object obj) {

		if (this == obj) {
			return true;
		}
		
		if (obj == null || !(obj instanceof HostPortPair)) {
			return false;
		}

		HostPortPair other = (HostPortPair) obj;

		if (!Utils.nullSafeEquals(this.host, other.host)) {
			return false;
		}

		if (!Utils.nullSafeEquals(this.port, other.port)) {
			return false;
		}

		return true;
	}

	@Override
	public String toString() {
		return host + ":" + port;
	}
	
	public String toStringWithProtocol(boolean ssl) {
		return (ssl ? PROTOCOL_REDISS : PROTOCOL_REDIS) + this.toString();
	}

	public static Set<String> convertToStringPairs(final Set<HostPortPair> pairs, boolean withProtocol, boolean... ssl) {
		
		if (pairs == null || pairs.isEmpty()) {
			return Collections.emptySet();
		}
		
		boolean sslToUse = (ssl != null && ssl.length > 0) ? ssl[0] : false;
		
		Set<String> textPairs = new LinkedHashSet<>(pairs.size());
		for (HostPortPair pair : pairs) {
			if (pair != null && !":".equals(pair.toString().trim())) {
				if (withProtocol) {
					textPairs.add((sslToUse ? PROTOCOL_REDISS : PROTOCOL_REDIS) + pair.toString());
				} else {
					textPairs.add(pair.toString());
				}

			}
		}
		return Collections.unmodifiableSet(textPairs);
	}

}
