/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.config;

import java.util.Collections;
import java.util.Set;

public interface RedisConfiguration {

	/**
	 * Default Configuration.
	 */
	RedisConfiguration defaulted = new RedisStandaloneConfiguration();

	/**
	 * Get the password to use.
	 *
	 * @return {@code null} by default.
	 */
	default String getPassword() {
		return null;
	}

	Set<String> getAddresses(boolean ssl);

	static boolean isSentinelConfiguration(RedisConfiguration configuration) {
		return configuration instanceof SentinelConfiguration;
	}

	static boolean isClusterConfiguration(RedisConfiguration configuration) {
		return configuration instanceof ClusterConfiguration;
	}
	
	interface DatabaseConfiguration {

		/**
		 * Get the database index to use.
		 *
		 * @return {@code zero} by default.
		 */
		default int getDatabase() {
			return 0;
		}
	}
	
	interface StandaloneConfiguration extends DatabaseConfiguration {

		String getHost();

		int getPort();

	}

	interface SentinelConfiguration extends DatabaseConfiguration {

		/**
		 * Get the {@literal Sentinel} master node.
		 *
		 * @return get the master node or {@literal null} if not set.
		 */
		String getMasterName();

		/**
		 * Returns an {@link Collections#unmodifiableSet(Set)} of {@literal Sentinels}.
		 *
		 * @return {@link Set} of sentinels. Never {@literal null}.
		 */
		Set<HostPortPair> getSentinels();
		
	}

	interface ClusterConfiguration {

		/**
		 * Returns an {@link Collections#unmodifiableSet(Set)} of {@literal cluster nodes}.
		 *
		 * @return {@link Set} of nodes. Never {@literal null}.
		 */
		Set<HostPortPair> getClusterNodes();

		/**
		 * @return max number of redirects to follow or {@literal null} if not set.
		 */
		int getMaxAttempts();

	}

}
