/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.config;

import static org.iherus.shiro.util.Utils.assertIsTrue;
import static org.iherus.shiro.util.Utils.assertNotBlank;

import java.util.Collections;
import java.util.Set;

import org.iherus.shiro.cache.redis.config.RedisConfiguration.StandaloneConfiguration;

public class RedisStandaloneConfiguration implements RedisConfiguration, StandaloneConfiguration {

	private static final String DEFAULT_HOST = "127.0.0.1";
	private static final int DEFAULT_PORT = 6379;

	private String host = DEFAULT_HOST;
	private int port = DEFAULT_PORT;
	private int database;
	private String password;

	/**
	 * Create a new default {@link RedisStandaloneConfiguration}.
	 */
	public RedisStandaloneConfiguration() {}

	/**
	 * Create a new {@link RedisStandaloneConfiguration} given {@code host}.
	 *
	 * @param host must not be {@literal null} or empty.
	 */
	public RedisStandaloneConfiguration(String host) {
		this(host, DEFAULT_PORT);
	}

	/**
	 * Create a new {@link RedisStandaloneConfiguration} given {@code host} and {@code port}.
	 *
	 * @param host must not be {@literal null} or empty.
	 * @param port     a valid TCP port (1-65535).
	 */
	public RedisStandaloneConfiguration(String host, int port) {
		assertNotBlank(host, "Host name must not be null or empty!");
		assertIsTrue(port >= 1 && port <= 65535,
				() -> String.format("Port %d must be a valid TCP port in the range between 1-65535!", port));

		this.host = host;
		this.port = port;
	}

	@Override
	public int getDatabase() {
		return database;
	}

	@Override
	public String getPassword() {
		return password;
	}

	@Override
	public String getHost() {
		return host;
	}

	public void setHost(String host) {
		this.host = host;
	}

	@Override
	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public void setDatabase(int database) {
		assertIsTrue(database >= 0, () -> String.format("Invalid DB index '%s' (a positive index required)", database));
		this.database = database;
	}

	public void setPassword(String password) {
		this.password = password;
	}
	
	public String getAddress(boolean ssl) {
		return new HostPortPair(host, port).toStringWithProtocol(ssl);
	}

	@Override
	public Set<String> getAddresses(boolean ssl) {
		return Collections.singleton(getAddress(ssl));
	}

}