/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

import io.lettuce.core.AbstractRedisClient;
import io.lettuce.core.RedisClient;
import io.lettuce.core.api.StatefulConnection;
import io.lettuce.core.cluster.RedisClusterClient;

/**
 * ConnectionProvider
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
@FunctionalInterface
public interface ConnectionProvider {

	/**
	 * @see RedisClient#connect()
	 * @see RedisClusterClient#connect()
	 */
	default <T extends StatefulConnection<?, ?>> T getConnection(Class<T> connectionType) {
		return Futures.join(getConnectionAsync(connectionType).toCompletableFuture());
	}
	
	/**
	 * @see RedisClient#connectAsync(io.lettuce.core.codec.RedisCodec, io.lettuce.core.RedisURI)
	 * @see RedisClusterClient#connectAsync(io.lettuce.core.codec.RedisCodec)
	 */
	<T extends StatefulConnection<?, ?>> CompletionStage<T> getConnectionAsync(Class<T> connectionType);

	/**
	 * @see StatefulConnection#close()
	 */
	default void release(StatefulConnection<?, ?> connection) {
		connection.close();
	}
	
	/**
	 * @see StatefulConnection#closeAsync()
	 */
	default CompletableFuture<Void> releaseAsync(StatefulConnection<?, ?> connection) {
		return connection.closeAsync();
	}
	
	
	interface ClientProvider {

		/**
		 * Returns the underlying Redis's Client.
		 *
		 * @return the {@link AbstractRedisClient}. Never {@literal null}.
		 */
		AbstractRedisClient getClient();
	}
	
}