/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import static org.iherus.shiro.cache.redis.Constant.GETDEL;
import static org.iherus.shiro.cache.redis.Constant.GETSET;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.util.RedisVerUtils;
import org.iherus.shiro.util.Utils;

import io.lettuce.core.KeyValue;
import io.lettuce.core.RedisURI;
import io.lettuce.core.ScriptOutputType;
import io.lettuce.core.api.sync.RedisCommands;
import io.lettuce.core.cluster.SlotHash;
import io.lettuce.core.cluster.api.StatefulRedisClusterConnection;
import io.lettuce.core.cluster.api.sync.RedisAdvancedClusterCommands;
import io.lettuce.core.cluster.models.partitions.RedisClusterNode;

/**
 * LettuceClusterConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class LettuceClusterConnection extends AbstractLettuceConnection implements RedisConnection {

	private final ConnectionPool pool;
	private final BatchOptions options;
	private final Duration timeout;

	private volatile StatefulRedisClusterConnection<byte[], byte[]> nativeConnection;

	private static final Function<byte[], Integer> calculator = ((key) -> {
		return SlotHash.getSlot(key);
	});

	public LettuceClusterConnection(ConnectionPool pool) {
		this(pool, BatchOptions.defaulted, RedisURI.DEFAULT_TIMEOUT_DURATION);
	}

	public LettuceClusterConnection(ConnectionPool pool, BatchOptions options, Duration timeout) {
		this.pool = pool;
		this.options = options;
		this.timeout = timeout;
	}

	public BatchOptions getOptions() {
		return options;
	}

	public Duration getTimeout() {
		return timeout;
	}

	@SuppressWarnings("unchecked")
	protected StatefulRedisClusterConnection<byte[], byte[]> getNativeConnection() {
		if (nativeConnection == null) {
			synchronized (this) {
				if (nativeConnection == null) {
					this.nativeConnection = this.pool.getConnection(StatefulRedisClusterConnection.class);
				}
			}
		}
		return nativeConnection;
	}

	protected RedisAdvancedClusterCommands<byte[], byte[]> getCommandExecutor() {
		RedisAdvancedClusterCommands<byte[], byte[]> commands = getNativeConnection().sync();
		commands.setTimeout(this.timeout);
		return commands;
	}

	@Override
	public byte[] get(byte[] key) {
		return getCommandExecutor().get(key);
	}

	@Override
	public byte[] set(byte[] key, byte[] value, Duration expired) {
		return getCommandExecutor().eval(GETSET, ScriptOutputType.VALUE, new byte[][] { key }, value,
				Utils.longToBytes(expired.toMillis()));
	}

	@Override
	public Long mdel(byte[]... keys) {
		if (Utils.isEmpty(keys)) return 0L;

		final RedisAdvancedClusterCommands<byte[], byte[]> commandExecutor = getCommandExecutor();

		boolean unlink = RedisVerUtils.getServerVersion(() -> {
			return parseServerVersion(commandExecutor.info("Server"));
		}).isSupportUnlink();

		return batchDeleteOnCluster(options.getDeleteBatchSize(), keys, ((batchKeys) -> {
			return unlink ? commandExecutor.unlink(batchKeys) : commandExecutor.del(batchKeys);
		}), calculator);
	}

	@Override
	public List<byte[]> mget(byte[]... keys) {
		final RedisAdvancedClusterCommands<byte[], byte[]> commandExecutor = getCommandExecutor();

		return batchGetOnCluster(options.getFetchBatchSize(), keys, (batchKeys) -> {
			List<KeyValue<byte[], byte[]>> kvs = commandExecutor.mget(batchKeys);
			return kvs.stream().map(kv -> kv.getValue()).collect(Collectors.toList());
		}, calculator);
	}

	@Override
	public byte[] del(byte[] key) {
		return getCommandExecutor().eval(GETDEL, ScriptOutputType.VALUE, key);
	}

	@Override
	public Set<byte[]> keys(byte[] pattern) {
		return distributionScanKeys((completion) -> {
			Map<RedisClusterNode, RedisCommands<byte[], byte[]>> masters = getCommandExecutor().masters().asMap();
			masters.forEach((node, commands) -> {
				completion.submit(() -> {
					return scanKeys(commands, pattern, this.options.getScanBatchSize());
				});
			});
			return masters.size();
		});
	}

	@Override
	public boolean isClusterConnection() {
		return true;
	}

	@Override
	public void close() {
		if (this.nativeConnection != null) {
			this.pool.release(this.nativeConnection);
		}
	}

}
