/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import static org.iherus.shiro.cache.redis.Constant.GETDEL;
import static org.iherus.shiro.cache.redis.Constant.GETSET;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.util.RedisVerUtils;
import org.iherus.shiro.util.Utils;

import io.lettuce.core.KeyValue;
import io.lettuce.core.RedisURI;
import io.lettuce.core.ScriptOutputType;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;

/**
 * LettuceConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class LettuceConnection extends AbstractLettuceConnection implements RedisConnection {

	private final ConnectionPool pool;
	private final BatchOptions options;
	private final Duration timeout;

	private Optional<Integer> database = Optional.empty();

	private volatile StatefulRedisConnection<byte[], byte[]> nativeConnection;

	public LettuceConnection(ConnectionPool pool) {
		this(pool, BatchOptions.defaulted, RedisURI.DEFAULT_TIMEOUT_DURATION);
	}

	public LettuceConnection(ConnectionPool pool, BatchOptions options, Duration timeout) {
		this.pool = pool;
		this.options = options;
		this.timeout = timeout;
	}

	public Optional<Integer> getDatabase() {
		return database;
	}

	public void setDatabase(Integer database) {
		this.database = Optional.ofNullable(database == null ? null : Math.max(0, database));
	}

	public BatchOptions getOptions() {
		return options;
	}

	public Duration getTimeout() {
		return timeout;
	}

	@SuppressWarnings("unchecked")
	public StatefulRedisConnection<byte[], byte[]> getNativeConnection() {
		if (nativeConnection == null) {
			synchronized (this) {
				if (nativeConnection == null) {
					this.nativeConnection = this.pool.getConnection(StatefulRedisConnection.class);
				}
			}
		}
		return nativeConnection;
	}

	protected RedisCommands<byte[], byte[]> getCommandExecutor() {
		StatefulRedisConnection<byte[], byte[]> connection = getNativeConnection();
		RedisCommands<byte[], byte[]> commands = connection.sync();
		commands.setTimeout(this.timeout);
		this.database.ifPresent(commands::select);
		return commands;
	}

	@Override
	public byte[] get(byte[] key) {
		return getCommandExecutor().get(key);
	}

	@Override
	public byte[] set(byte[] key, byte[] value, Duration expired) {
		return getCommandExecutor().eval(GETSET, ScriptOutputType.VALUE, new byte[][] { key }, value,
				Utils.longToBytes(expired.toMillis()));
	}

	@Override
	public Long mdel(byte[]... keys) {
		if (Utils.isEmpty(keys)) return 0L;

		boolean unlink = RedisVerUtils.getServerVersion(() -> {
			return parseServerVersion(getCommandExecutor().info("Server"));
		}).isSupportUnlink();

		return batchDeleteOnStandalone(options.getDeleteBatchSize(), keys, (batchKeys) -> {
			return unlink ? getCommandExecutor().unlink(batchKeys) : getCommandExecutor().del(batchKeys);
		});
	}

	@Override
	public List<byte[]> mget(byte[]... keys) {
		return batchGetOnStandalone(options.getFetchBatchSize(), keys, (batchKeys) -> {
			List<KeyValue<byte[], byte[]>> kvs = getCommandExecutor().mget(batchKeys);
			return kvs.stream().map(kv -> kv.getValue()).collect(Collectors.toList());
		});
	}

	@Override
	public byte[] del(byte[] key) {
		return getCommandExecutor().eval(GETDEL, ScriptOutputType.VALUE, key);
	}

	@Override
	public Set<byte[]> keys(byte[] pattern) {
		return scanKeys(getCommandExecutor(), pattern, options.getScanBatchSize());
	}

	@Override
	public void close() {
		if (this.nativeConnection != null) {
			this.pool.release(this.nativeConnection);
		}
	}

}
