/**
 * Copyright (c) 2016-2019, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.spring;

import static org.iherus.shiro.util.Utils.assertNotNull;

import java.util.Optional;

import org.iherus.shiro.cache.redis.config.RedisConfiguration;
import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.cache.redis.connection.RedisConnectionFactory;

/**
 * Implemention for compatibility with
 * {@link org.springframework.data.redis.connection.RedisConnectionFactory}
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class CompatibleRedisConnectionFactory implements RedisConnectionFactory {

	private org.springframework.data.redis.connection.RedisConnectionFactory nativeConnectionFactory;

	private Optional<BatchOptions> batchOptions = Optional.empty();

	public CompatibleRedisConnectionFactory() {}

	public CompatibleRedisConnectionFactory(
			org.springframework.data.redis.connection.RedisConnectionFactory connectionFactory) {
		assertNotNull(connectionFactory, "Spring's RedisConnectionFactory must not be null.");
		this.nativeConnectionFactory = connectionFactory;
	}

	public Optional<BatchOptions> getBatchOptions() {
		return batchOptions;
	}

	public void setBatchOptions(BatchOptions batchOptions) {
		this.batchOptions = Optional.ofNullable(batchOptions);
	}

	public org.springframework.data.redis.connection.RedisConnectionFactory getNativeConnectionFactory() {
		return nativeConnectionFactory;
	}

	public void setNativeConnectionFactory(
			org.springframework.data.redis.connection.RedisConnectionFactory connectionFactory) {
		this.nativeConnectionFactory = connectionFactory;
	}

	@Override
	public RedisConnection getConnection() {
		return new CompatibleRedisConnection(this.nativeConnectionFactory.getConnection(),
				this.batchOptions.orElse(BatchOptions.defaulted));
	}

	@Override
	public RedisConfiguration getConfiguration() {
		throw new UnsupportedOperationException("This operation is not supported.");
	}

}
