/*
 * Decompiled with CFR 0.152.
 */
package org.iherus.shiro.cache.redis;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.shiro.cache.Cache;
import org.apache.shiro.cache.CacheException;
import org.iherus.shiro.cache.redis.Constant;
import org.iherus.shiro.cache.redis.ExpiredCache;
import org.iherus.shiro.cache.redis.RedisOperations;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.cache.redis.serializer.StringSerializer;
import org.iherus.shiro.util.Md5Utils;
import org.iherus.shiro.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RedisCache<K, V>
implements Cache<K, V>,
ExpiredCache<K, V>,
ExpiredCache.Named {
    private static final Logger logger = LoggerFactory.getLogger(RedisCache.class);
    private final String name;
    private final RedisOperations operations;
    private String keyPrefix;
    private Duration expiration;
    private Optional<Integer> database;

    public RedisCache(String name, RedisOperations operations) {
        Utils.assertNotBlank(name, "Name must not be blank.");
        Utils.assertNotNull(operations, "RedisOperations must not be null.");
        this.name = name;
        this.operations = operations;
        this.database = Optional.empty();
    }

    public RedisCache(String name, RedisOperations operations, String keyPrefix, Duration expiration, Integer database) {
        this(name, operations);
        this.keyPrefix = keyPrefix;
        this.expiration = expiration;
        this.database = Optional.ofNullable(database);
    }

    @Override
    public String getName() {
        return this.name;
    }

    public RedisOperations getOperations() {
        return this.operations;
    }

    public String getKeyPrefix() {
        return this.keyPrefix;
    }

    public void setKeyPrefix(String keyPrefix) {
        this.keyPrefix = keyPrefix;
    }

    public Duration getExpiration() {
        return this.expiration;
    }

    public void setExpiration(Duration expiration) {
        this.expiration = expiration;
    }

    public Optional<Integer> getDatabase() {
        return this.database;
    }

    public void setDatabase(Integer database) {
        this.database = Optional.ofNullable(database);
    }

    protected void selectDatabase(RedisConnection connection) {
        this.database.filter(db -> db >= 0 && !connection.isClusterConnection()).ifPresent(it -> Utils.invokeMethod((Object)connection, "setDatabase", Integer.class, it));
    }

    public V get(K key) throws CacheException {
        if (logger.isDebugEnabled()) {
            logger.debug("Getting object from cache [" + this.getName() + "] for key [" + key + "]");
        }
        if (key == null) {
            return null;
        }
        try {
            return (V)this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                byte[] value = connection.get(this.getKeyToBytes(key));
                if (logger.isDebugEnabled() && !Utils.isEmpty(value)) {
                    logger.debug("Cache for [" + key + "] is exist, ready to deserialize it.");
                }
                return this.deserializeValue(value);
            });
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    public V put(K key, V value) throws CacheException {
        return this.put(key, value, null);
    }

    @Override
    public V put(K key, V value, Duration expired) throws CacheException {
        if (logger.isDebugEnabled()) {
            logger.debug("Putting object in cache [" + this.getName() + "] for key [" + key + "]");
        }
        if (key == null) {
            return value;
        }
        try {
            return (V)this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                Duration expirationToUse = expired != null ? expired : (this.getExpiration() == null ? Constant.DEFAULT_CACHE_EXPIRATION : this.getExpiration());
                byte[] previous = connection.set(this.getKeyToBytes(key), this.serializeValue(value), expirationToUse);
                return this.deserializeValue(previous);
            });
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    public V remove(K key) throws CacheException {
        if (logger.isDebugEnabled()) {
            logger.debug("Removing object from cache [" + this.getName() + "] for key [" + key + "]");
        }
        if (key == null) {
            return null;
        }
        try {
            return (V)this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                byte[] previous = connection.del(this.getKeyToBytes(key));
                if (logger.isDebugEnabled() && !Utils.isEmpty(previous)) {
                    logger.debug("Remove key [{}] successfully.", key);
                }
                return this.deserializeValue(previous);
            });
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    public void clear() throws CacheException {
        if (logger.isDebugEnabled()) {
            logger.debug("Clear all cached objects.");
        }
        try {
            this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                Set<byte[]> allKeys = connection.keys(StringSerializer.UTF_8.serialize(this.getKeyPrefix() + "*"));
                if (logger.isDebugEnabled() && !allKeys.isEmpty()) {
                    logger.debug("Currently scanning to {} keys, ready to clear.", (Object)allKeys.size());
                }
                Long c = connection.mdel((byte[][])allKeys.toArray((T[])new byte[allKeys.size()][]));
                if (logger.isDebugEnabled() && c != null) {
                    logger.debug("After the cleanup is completed, this task clears {} cache objects.", (Object)c);
                }
                return c;
            });
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    public int size() {
        try {
            return this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                return connection.size(StringSerializer.UTF_8.serialize(this.getKeyPrefix() + "*"));
            }).intValue();
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    public Set<K> keys() {
        try {
            return this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                Set<byte[]> allKeys = connection.keys(StringSerializer.UTF_8.serialize(this.getKeyPrefix() + "*"));
                if (logger.isDebugEnabled() && !allKeys.isEmpty()) {
                    logger.debug("Currently scanning to {} keys.", (Object)allKeys.size());
                }
                Set result = allKeys.stream().map(key -> this.deserializeKey((byte[])key)).collect(Collectors.toSet());
                return Collections.unmodifiableSet(result);
            });
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    public Collection<V> values() {
        try {
            return this.operations.execute(connection -> {
                this.selectDatabase((RedisConnection)connection);
                Set<byte[]> allKeys = connection.keys(StringSerializer.UTF_8.serialize(this.getKeyPrefix() + "*"));
                List<byte[]> allValues = connection.mget((byte[][])allKeys.toArray((T[])new byte[allKeys.size()][]));
                if (logger.isDebugEnabled() && !allValues.isEmpty()) {
                    logger.debug("Currently scanning to {} key-values.", (Object)allValues.size());
                }
                List result = allValues.stream().map(it -> this.deserializeValue((byte[])it)).collect(Collectors.toList());
                return Collections.unmodifiableList(result);
            });
        }
        catch (Exception e) {
            throw new CacheException((Throwable)e);
        }
    }

    protected byte[] getKeyToBytes(K key) {
        return this.getKeyToBytes(key, this.getKeyPrefix());
    }

    private byte[] getKeyToBytes(K key, String keyPrefix) {
        byte[] keyBytes = new byte[]{};
        if (key instanceof byte[]) {
            keyBytes = (byte[])key;
        } else if (key instanceof String) {
            keyBytes = this.serializeKey(key.toString().replace(keyPrefix, ""));
        } else {
            if (logger.isWarnEnabled()) {
                logger.warn("The current cache key is not byte[] or String type, and the key should be kept unchanged in subsequent operations.");
            }
            byte[] obytes = this.serializeValue(key);
            String keyToUse = Md5Utils.getMd5(obytes, key.getClass().getSimpleName()) + new String(obytes).hashCode();
            keyBytes = this.serializeKey(keyToUse);
        }
        return Utils.mergeAll(this.serializeKey(keyPrefix), keyBytes);
    }

    private byte[] serializeKey(String key) {
        return this.operations.getKeySerializer().serialize((Object)key);
    }

    private String deserializeKey(byte[] bytes) {
        if (Utils.isEmpty(bytes)) {
            return null;
        }
        return (String)this.operations.getKeySerializer().deserialize(bytes);
    }

    private byte[] serializeValue(Object value) {
        return this.operations.getValueSerializer().serialize(value);
    }

    private Object deserializeValue(byte[] bytes) {
        if (Utils.isEmpty(bytes)) {
            return null;
        }
        return this.operations.getValueSerializer().deserialize(bytes);
    }
}

