/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis;

import org.apache.shiro.cache.Cache;
import org.apache.shiro.cache.CacheManager;
import org.apache.shiro.session.Session;
import org.apache.shiro.session.mgt.eis.CachingSessionDAO;

import java.io.Serializable;
import java.time.Duration;

/**
 * Implementation of {@link org.apache.shiro.session.mgt.eis.SessionDAO},
 * it based on memory cache to alleviate {@literal Redis} access pressure.
 *
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class RedisSessionDAO extends CachingSessionDAO {

    /**
     * L1Cache instance.
     */
    private Cache<Serializable, Session> l1Cache;

    /**
     * Enable L1Cache to reduce the frequency of redis access.
     */
    private boolean l1CacheEnabled;

    public RedisSessionDAO(CacheManager cacheManager) {
        setCacheManager(cacheManager);
        this.l1CacheEnabled = true;
        this.l1Cache = new MemorySessionCache();
    }

    public Cache<Serializable, Session> getL1Cache() {
        return l1Cache;
    }

    public void setL1Cache(Cache<Serializable, Session> l1Cache) {
        this.l1Cache = l1Cache;
    }

    public boolean isL1CacheEnabled() {
        return l1CacheEnabled && (l1Cache != null);
    }

    public void setL1CacheEnabled(boolean l1CacheEnabled) {
        this.l1CacheEnabled = l1CacheEnabled;
    }

    protected boolean isExpiredCacheAware(Cache cache) {
        return cache instanceof ExpiredCache;
    }

    @Override
    protected void cache(Session session, Serializable sessionId, Cache<Serializable, Session> cache) {
        if (!isExpiredCacheAware(cache)) {
            super.cache(session, sessionId, cache);
        } else {
            // A negative return value (0 or lesser) means the session will never expire.
            Duration expiration = session.getTimeout() <= 0l ? Duration.ZERO : Duration.ofMillis(session.getTimeout());
            ((ExpiredCache) cache).put(sessionId, session, expiration);
        }

        if (isL1CacheEnabled()) {
            this.l1Cache.put(sessionId, session);
        }
    }

    @Override
    protected void doUpdate(Session session) {
        this.doDelete(session);
    }

    @Override
    protected void doDelete(Session session) {
        if (session != null && session.getId() != null && isL1CacheEnabled()) {
            this.l1Cache.remove(session.getId());
        }
    }

    @Override
    protected Serializable doCreate(Session session) {
        Serializable sessionId = generateSessionId(session);
        assignSessionId(session, sessionId);
        return sessionId;
    }

    @Override
    protected Session getCachedSession(Serializable sessionId, Cache<Serializable, Session> cache) {
        if (isL1CacheEnabled()) {
            Session session = this.l1Cache.get(sessionId);
            if (session == null) {
                session = super.getCachedSession(sessionId, cache);
                if (session != null) {
                    this.l1Cache.put(sessionId, session);
                }
            }
            return session;
        }

        return super.getCachedSession(sessionId, cache);
    }

    /**
     * {@link CachingSessionDAO#readSession(Serializable)}
     */
    @Override
    protected Session doReadSession(Serializable sessionId) {
        return null; //should never execute because this implementation relies on parent class to access cache, which
        //is where all sessions reside - it is the cache implementation that determines if the
        //cache is memory only or disk-persistent, etc.
    }

}
