/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.config;

import static org.iherus.shiro.util.Utils.assertIsTrue;
import static org.iherus.shiro.util.Utils.assertNotNull;
import static org.iherus.shiro.util.Utils.commaDelimitedListToSet;
import static org.iherus.shiro.util.Utils.isNotBlank;
import static org.iherus.shiro.util.Utils.split;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

import org.iherus.shiro.cache.redis.config.RedisConfiguration.SentinelConfiguration;

public class RedisSentinelConfiguration implements RedisConfiguration, SentinelConfiguration {

	private String masterName;
	private Set<HostPortPair> sentinels;
	private int database;
	private String password;

	public RedisSentinelConfiguration() {
		this.sentinels = new LinkedHashSet<HostPortPair>();
	}

	@Override
	public String getMasterName() {
		return masterName;
	}

	public void setMasterName(String masterName) {
		this.masterName = masterName;
	}

	@Override
	public int getDatabase() {
		return database;
	}

	public void setDatabase(int database) {
		assertIsTrue(database >= 0, () -> String.format("Invalid DB index '%s' (a positive index required)", database));
		this.database = database;
	}

	@Override
	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	@Override
	public Set<HostPortPair> getSentinels() {
		return Collections.unmodifiableSet(sentinels);
	}

	public void setSentinels(Set<HostPortPair> sentinels) {
		this.sentinels = sentinels;
	}

	public void setSentinelsFromText(String sentinels) {
		String sentinelsToUse = isNotBlank(sentinels) ? sentinels.replace(" ", "") : sentinels;
		addSentinels(commaDelimitedListToSet(sentinelsToUse));
	}

	/**
	 * Add sentinel.
	 *
	 * @param sentinel must not be {@literal null}.
	 */
	public void addSentinel(HostPortPair sentinel) {
		assertNotNull(sentinel, "Sentinel must not be 'null'.");
		this.sentinels.add(sentinel);
	}

	private void addSentinels(Set<String> hostAndPorts) {
		for (String hostAndPort : hostAndPorts) {
			addSentinel(readHostAndPortFromString(hostAndPort));
		}
	}

	private HostPortPair readHostAndPortFromString(String hostAndPort) {
		String[] args = split(hostAndPort, ":");
		assertNotNull(args, "HostAndPort need to be seperated by  ':'.");
		assertIsTrue(args.length == 2, () -> "Host and Port String needs to specified as host:port");
		return new HostPortPair(args[0], Integer.valueOf(args[1]).intValue());
	}
	
	public Set<String> getTextSentinels() {
		return HostPortPair.convertToStringPairs(getSentinels(), false);
	}

	@Override
	public Set<String> getAddresses(boolean ssl) {
		return HostPortPair.convertToStringPairs(getSentinels(), true, ssl);
	}

}