/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.jedis;

import static org.iherus.shiro.cache.redis.Constant.GETDEL;
import static org.iherus.shiro.cache.redis.Constant.GETSET;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.util.RedisVerUtils;
import org.iherus.shiro.util.Utils;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.util.JedisClusterCRC16;
import redis.clients.jedis.util.SafeEncoder;

/**
 * JedisClusterConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class JedisClusterConnection extends AbstractJedisConnection implements RedisConnection {

	private final JedisCluster cluster;
	private final BatchOptions options;

	private static final Function<byte[], Integer> calculator = ((key) -> {
		return JedisClusterCRC16.getSlot(key);
	});

	public JedisClusterConnection(JedisCluster cluster) {
		this(cluster, BatchOptions.defaulted);
	}

	public JedisClusterConnection(JedisCluster cluster, BatchOptions options) {
		this.cluster = cluster;
		this.options = options;
	}

	public JedisCluster getCluster() {
		return cluster;
	}

	public BatchOptions getOptions() {
		return options;
	}

	@Override
	public byte[] get(byte[] key) {
		return getCluster().get(key);
	}

	@Override
	public byte[] set(byte[] key, byte[] value, Duration expired) {
		List<byte[]> args = Arrays.asList(value, Utils.longToBytes(Duration.ZERO.equals(expired) ? -1l : expired.toMillis()));
		return (byte[]) getCluster().eval(SafeEncoder.encode(GETSET), Collections.singletonList(key), args);
	}

	@Override
	public Long mdel(byte[]... keys) {
		
		if (Utils.isEmpty(keys)) return 0L;

		boolean unlink = RedisVerUtils.getServerVersion(() -> {
			JedisPool pool = getCluster().getClusterNodes().values().iterator().next();
			return parseServerVersion(pool.getResource().info("Server"));
		}).isSupportUnlink();

		return batchDeleteOnCluster(options.getDeleteBatchSize(), keys, ((batchKeys) -> {
			return unlink ? getCluster().unlink(batchKeys) : getCluster().del(batchKeys);
		}), calculator);
	}
	
	@Override
	public List<byte[]> mget(byte[]... keys) {
		return batchGetOnCluster(options.getFetchBatchSize(), keys, (batchKeys) -> {
			return getCluster().mget(batchKeys);
		}, calculator);
	}

	@Override
	public byte[] del(byte[] key) {
		return (byte[]) getCluster().eval(SafeEncoder.encode(GETDEL), 1, key);
	}

	@Override
	public Set<byte[]> keys(byte[] pattern) {
		return distributionScanKeys((completion) -> {
			Map<String, JedisPool> nodes = getCluster().getClusterNodes();
			nodes.forEach((key, pool) -> {
				completion.submit(() -> {
					return getKeysFromNode(pool, pattern);
				});
			});

			return nodes.size();
		});
	}

	@Override
	public boolean isClusterConnection() {
		return true;
	}

	private Set<byte[]> getKeysFromNode(JedisPool pool, byte[] pattern) {
		Jedis connection = null;
		try {
			connection = pool.getResource();
			if (!isMaster(connection)) {
				return Collections.emptySet();
			}
			Set<byte[]> keys = scanKeys(connection, pattern, options.getScanBatchSize());
			return keys;
		} finally {
			releaseConnection(connection);
		}
	}

	@Override
	public void close() {
		// TODO Auto-generated method stub
	}

}
