/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.jedis;

import static org.iherus.shiro.util.Utils.assertNotNull;

import java.time.Duration;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.SSLSocketFactory;

import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.iherus.shiro.cache.redis.config.DefaultPoolConfig;
import org.iherus.shiro.cache.redis.config.HostPortPair;
import org.iherus.shiro.cache.redis.config.RedisClusterConfiguration;
import org.iherus.shiro.cache.redis.config.RedisConfiguration;
import org.iherus.shiro.cache.redis.config.RedisSentinelConfiguration;
import org.iherus.shiro.cache.redis.config.RedisStandaloneConfiguration;
import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.Destroyable;
import org.iherus.shiro.cache.redis.connection.Initializable;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.cache.redis.connection.RedisConnectionFactory;
import org.iherus.shiro.exception.InvocationException;

import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisSentinelPool;
import redis.clients.jedis.Protocol;
import redis.clients.jedis.util.Pool;

/**
 * Implementation of {@link RedisConnectionFactory} a based on {@literal Jedis}.
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class JedisConnectionFactory implements RedisConnectionFactory, Initializable, Destroyable {

	private volatile Pool<Jedis> pool;
	private volatile JedisCluster cluster;

	private RedisConfiguration configuration = RedisConfiguration.defaulted;
	private GenericObjectPoolConfig<?> poolConfig;

	private String clientName;
	private Duration soTimeout = Duration.ofMillis(Protocol.DEFAULT_TIMEOUT);
	private Duration connectTimeout = Duration.ofMillis(Protocol.DEFAULT_TIMEOUT);

	private boolean useSsl;
	private SSLSocketFactory sslSocketFactory;
	private SSLParameters sslParameters;
	private HostnameVerifier hostnameVerifier;

	private Optional<BatchOptions> batchOptions = Optional.empty();
	
	private final Object lock = new Object();

	public JedisConnectionFactory() {
		this(new DefaultPoolConfig());
	}

	public JedisConnectionFactory(GenericObjectPoolConfig<?> poolConfig) {
		this.poolConfig = poolConfig == null ? new DefaultPoolConfig() : poolConfig;
	}

	public JedisConnectionFactory(RedisStandaloneConfiguration standaloneConfig) {
		this(standaloneConfig, new DefaultPoolConfig());
	}

	public JedisConnectionFactory(RedisSentinelConfiguration sentinelConfig) {
		this(sentinelConfig, new DefaultPoolConfig());
	}

	public JedisConnectionFactory(RedisClusterConfiguration clusterConfig) {
		this(clusterConfig, new DefaultPoolConfig());
	}

	public JedisConnectionFactory(RedisStandaloneConfiguration standaloneConfig,
			GenericObjectPoolConfig<?> poolConfig) {
		this(poolConfig);
		assertNotNull(standaloneConfig, "RedisStandaloneConfiguration must not be null.");
		this.configuration = standaloneConfig;
	}

	public JedisConnectionFactory(RedisSentinelConfiguration sentinelConfig, GenericObjectPoolConfig<?> poolConfig) {
		this(poolConfig);
		assertNotNull(sentinelConfig, "RedisSentinelConfiguration must not be null.");
		this.configuration = sentinelConfig;
	}

	public JedisConnectionFactory(RedisClusterConfiguration clusterConfig, GenericObjectPoolConfig<?> poolConfig) {
		this(poolConfig);
		assertNotNull(clusterConfig, "RedisClusterConfiguration must not be null.");
		this.configuration = clusterConfig;
	}

	protected Pool<Jedis> getPool() {
		if (pool == null) {
			synchronized (lock) {
				if (pool == null) {
					this.pool = createPool();
				}
			}
		}
		return pool;
	}

	protected JedisCluster getCluster() {
		if (cluster == null) {
			synchronized (lock) {
				if (cluster == null) {
					this.cluster = createCluster();
				}
			}
		}
		return cluster;
	}

	public RedisConfiguration getConfiguration() {
		return configuration;
	}

	public void setConfiguration(RedisConfiguration configuration) {
		this.configuration = configuration;
	}

	public GenericObjectPoolConfig<?> getPoolConfig() {
		return poolConfig;
	}

	public void setPoolConfig(GenericObjectPoolConfig<?> poolConfig) {
		this.poolConfig = poolConfig;
	}

	public String getClientName() {
		return clientName;
	}

	public void setClientName(String clientName) {
		this.clientName = clientName;
	}

	public Duration getSoTimeout() {
		return soTimeout;
	}

	public void setSoTimeout(Duration soTimeout) {
		this.soTimeout = soTimeout;
	}
	
	public void setSoTimeoutMillis(long soTimeoutMillis) {
		this.setSoTimeout(Duration.ofMillis(soTimeoutMillis));
	}

	public Duration getConnectTimeout() {
		return connectTimeout;
	}

	public void setConnectTimeout(Duration connectTimeout) {
		this.connectTimeout = connectTimeout;
	}

	public void setConnectTimeoutMillis(long connectTimeoutMillis) {
		this.setConnectTimeout(Duration.ofMillis(connectTimeoutMillis));
	}
	
	protected int getSoTimeoutAsMillis() {
		return Math.toIntExact(getSoTimeout().toMillis());
	}

	protected int getConnectionTimeoutAsMillis() {
		return Math.toIntExact(getConnectTimeout().toMillis());
	}

	public boolean isUseSsl() {
		return useSsl;
	}

	public void setUseSsl(boolean useSsl) {
		this.useSsl = useSsl;
	}

	public SSLSocketFactory getSslSocketFactory() {
		return sslSocketFactory;
	}

	public void setSslSocketFactory(SSLSocketFactory sslSocketFactory) {
		this.sslSocketFactory = sslSocketFactory;
	}

	public SSLParameters getSslParameters() {
		return sslParameters;
	}

	public void setSslParameters(SSLParameters sslParameters) {
		this.sslParameters = sslParameters;
	}

	public HostnameVerifier getHostnameVerifier() {
		return hostnameVerifier;
	}

	public void setHostnameVerifier(HostnameVerifier hostnameVerifier) {
		this.hostnameVerifier = hostnameVerifier;
	}

	public Optional<BatchOptions> getBatchOptions() {
		return batchOptions;
	}

	public void setBatchOptions(BatchOptions options) {
		this.batchOptions = Optional.ofNullable(options);
	}

	@Override
	public RedisConnection getConnection() {
		if (isRedisClusterAware()) {
			return getClusterConnection();
		}
		Jedis jedis = fetchJedisConnection();
		return new JedisConnection(jedis, batchOptions.orElse(BatchOptions.defaulted));
	}

	public JedisClusterConnection getClusterConnection() {
		if (!isRedisClusterAware()) {
			throw new InvocationException("Cluster is not configured.");
		}
		return new JedisClusterConnection(getCluster(), batchOptions.orElse(BatchOptions.defaulted));
	}

	public boolean isRedisSentinelAware() {
		return RedisConfiguration.isSentinelConfiguration(configuration);
	}

	public boolean isRedisClusterAware() {
		return RedisConfiguration.isClusterConfiguration(configuration);
	}

	/**
	 * Returns a Jedis instance to be used as a Redis connection.
	 */
	protected Jedis fetchJedisConnection() {
		return getPool().getResource();
	}

	private Pool<Jedis> createPool() {
		if (isRedisSentinelAware()) {
			return createRedisSentinelPool((RedisSentinelConfiguration) this.configuration);
		}
		return createRedisStandalonePool((RedisStandaloneConfiguration) this.configuration);
	}

	protected Pool<Jedis> createRedisStandalonePool(RedisStandaloneConfiguration config) {
		assertNotNull(config, "RedisStandaloneConfiguration must not be null.");
		return new JedisPool(getPoolConfig(), config.getHost(), config.getPort(), getConnectionTimeoutAsMillis(),
				getSoTimeoutAsMillis(), config.getPassword(), config.getDatabase(), getClientName(), isUseSsl(),
				getSslSocketFactory(), getSslParameters(), getHostnameVerifier());
	}

	protected Pool<Jedis> createRedisSentinelPool(RedisSentinelConfiguration config) {
		assertNotNull(config, "RedisSentinelConfiguration must not be null.");
		return new JedisSentinelPool(config.getMasterName(), config.getTextSentinels(), getPoolConfig(),
				getConnectionTimeoutAsMillis(), getSoTimeoutAsMillis(), config.getPassword(), config.getDatabase(),
				getClientName());
	}

	private JedisCluster createCluster() {
		return createRedisCluster((RedisClusterConfiguration) this.configuration);
	}

	protected JedisCluster createRedisCluster(RedisClusterConfiguration config) {
		assertNotNull(config, "RedisClusterConfiguration must not be null.");
		Set<HostAndPort> clusterNodes = new HashSet<>();
		for (HostPortPair node : config.getClusterNodes()) {
			clusterNodes.add(new HostAndPort(node.getHost(), node.getPort()));
		}
		return new JedisCluster(clusterNodes, getConnectionTimeoutAsMillis(), getSoTimeoutAsMillis(),
				config.getMaxAttempts(), config.getPassword(), getClientName(), getPoolConfig(), isUseSsl(),
				getSslSocketFactory(), getSslParameters(), getHostnameVerifier(), null);
	}

	@Override
	public void init() throws Exception {
		if (!isRedisClusterAware() && this.pool == null) { // Standalone or Sentinel
			this.pool = createPool();
		}

		if (isRedisClusterAware() && this.cluster == null) { // Cluster
			this.cluster = createCluster();
		}
	}

	@Override
	public void destroy() throws Exception {
		try {
			if (this.pool != null) this.pool.close();
			if (this.cluster != null) this.cluster.close();
		} finally {
			this.pool = null;
			this.cluster = null;
		}
	}

}
