/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import static org.iherus.shiro.util.Utils.assertNotNull;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import org.apache.commons.pool2.BasePooledObjectFactory;
import org.apache.commons.pool2.PooledObject;
import org.apache.commons.pool2.impl.DefaultPooledObject;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.iherus.shiro.cache.redis.connection.Destroyable;
import org.iherus.shiro.cache.redis.connection.lettuce.StandaloneConnectionProvider.DatabaseProvider;
import org.iherus.shiro.exception.ConnectionPoolException;
import org.iherus.shiro.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.lettuce.core.AbstractRedisClient;
import io.lettuce.core.api.StatefulConnection;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.support.AsyncConnectionPoolSupport;
import io.lettuce.core.support.AsyncPool;
import io.lettuce.core.support.BoundedPoolConfig;
import io.lettuce.core.support.CommonsPool2ConfigConverter;

/**
 * Lettuce_Commons-pool2 ConnectionPool
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public abstract class ConnectionPool implements ConnectionProvider, Destroyable {

	@Override
	public abstract <T extends StatefulConnection<?, ?>> T getConnection(Class<T> connectionType);

	@Override
	public abstract <T extends StatefulConnection<?, ?>> CompletionStage<T> getConnectionAsync(Class<T> connectionType);

	@Override
	public abstract void release(StatefulConnection<?, ?> connection);

	@Override
	public abstract CompletableFuture<Void> releaseAsync(StatefulConnection<?, ?> connection);

	/**
	 * {@link ConnectionProvider} with connection pooling support. This connection provider holds multiple pools (one
	 * per connection type and allocation type (synchronous/asynchronous)) for contextualized connection allocation.
	 * <p />
	 * Each allocated connection is tracked and to be returned into the pool which created the connection. Instances of this
	 * class require {@link #destroy() disposal} to de-allocate lingering connections that were not returned to the pool and
	 * to close the pools.
	 * <p />
	 * This provider maintains separate pools due to the allocation nature (synchronous/asynchronous). Asynchronous
	 * connection pooling requires a non-blocking allocation API. Connections requested asynchronously can be returned
	 * synchronously and vice versa. A connection obtained synchronously is returned to the synchronous pool even if
	 * {@link #releaseAsync(StatefulConnection) released asynchronously}. This is an undesired case as the synchronous pool
	 * will block the asynchronous flow for the time of release.
	 *
	 * @author Mark Paluch
	 * @author Christoph Strobl
	 * 
	 * @modifier Bosco.Liao
	 * @indicate Copy from {@link org.springframework.data.redis.connection.lettuce.LettucePoolingConnectionProvider} 
	 */
	static class LettuceSmartConnectionPool extends ConnectionPool implements ClientProvider {

		private final static Logger logger = LoggerFactory.getLogger(LettuceSmartConnectionPool.class);

		private final ConnectionProvider nativeConnectionProvider;

		@SuppressWarnings("rawtypes")
		private final GenericObjectPoolConfig poolConfig;
		private final BoundedPoolConfig asyncPoolConfig;

		private final Map<StatefulConnection<?, ?>, GenericObjectPool<StatefulConnection<?, ?>>> poolRef = new ConcurrentHashMap<>(32);
		private final Map<StatefulConnection<?, ?>, AsyncPool<StatefulConnection<?, ?>>> asyncPoolRef = new ConcurrentHashMap<>(32);
		private final Map<CompletableFuture<StatefulConnection<?, ?>>, AsyncPool<StatefulConnection<?, ?>>> inProgressAsyncPoolRef = new ConcurrentHashMap<>(32);
		
		private final Map<Class<?>, GenericObjectPool<StatefulConnection<?, ?>>> pools = new ConcurrentHashMap<>(32);
		private final Map<Class<?>, AsyncPool<StatefulConnection<?, ?>>> asyncPools = new ConcurrentHashMap<>(32);

		@SuppressWarnings("rawtypes")
		LettuceSmartConnectionPool(ConnectionProvider connectionProvider, GenericObjectPoolConfig poolConfig) {
			assertNotNull(connectionProvider, "Native connection-provider must not be null.");
			assertNotNull(poolConfig, "GenericObjectPoolConfig must not be null.");

			this.nativeConnectionProvider = connectionProvider;
			this.poolConfig = poolConfig;
			this.asyncPoolConfig = CommonsPool2ConfigConverter.bounded(this.poolConfig);
		}

		@Override
		public <T extends StatefulConnection<?, ?>> T getConnection(Class<T> connectionType) {
			@SuppressWarnings("unchecked")
			GenericObjectPool<StatefulConnection<?, ?>> pool = pools.computeIfAbsent(connectionType, poolType -> {
				return (GenericObjectPool<StatefulConnection<?, ?>>) new GenericObjectPool<T>(
						new LettucePooledObjectFactory<T>(nativeConnectionProvider, () -> connectionType), poolConfig);

			});
			try {
				StatefulConnection<?, ?> connection = pool.borrowObject();
				poolRef.put(connection, pool);
				return connectionType.cast(connection);
			} catch (Exception e) {
				throw new ConnectionPoolException("Could not get a resource from the pool", e);
			}
		}
		
		@Override
		public <T extends StatefulConnection<?, ?>> CompletionStage<T> getConnectionAsync(Class<T> connectionType) {
			
			AsyncPool<StatefulConnection<?, ?>> pool = asyncPools.computeIfAbsent(connectionType, poolType -> {
				return AsyncConnectionPoolSupport.createBoundedObjectPool(() -> nativeConnectionProvider.getConnectionAsync(connectionType) //
								.thenApply(connectionType::cast), asyncPoolConfig, false);
			});

			CompletableFuture<StatefulConnection<?, ?>> acquire = pool.acquire();

			inProgressAsyncPoolRef.put(acquire, pool);
			
			return acquire.whenComplete((connection, e) -> {

				inProgressAsyncPoolRef.remove(acquire);

				if (connection != null) {
					asyncPoolRef.put(connection, pool);
				}
			}).thenApply(connectionType::cast);
			
		}

		@Override
		public void release(StatefulConnection<?, ?> connection) {
			
			GenericObjectPool<StatefulConnection<?, ?>> pool = poolRef.remove(connection);

			if (pool == null) {

				AsyncPool<StatefulConnection<?, ?>> asyncPool = asyncPoolRef.remove(connection);

				if (asyncPool == null) {
					throw new ConnectionPoolException("Returned connection " + connection
							+ " was either previously returned or does not belong to this connection provider");
				}

				discardIfNecessary(connection);
				asyncPool.release(connection).join();
				return;
			}

			discardIfNecessary(connection);
			pool.returnObject(connection);
		}

		private void discardIfNecessary(StatefulConnection<?, ?> connection) {

			if (connection instanceof StatefulRedisConnection) {

				StatefulRedisConnection<?, ?> redisConnection = (StatefulRedisConnection<?, ?>) connection;
				if (redisConnection.isMulti()) {
					redisConnection.async().discard();
				}
			}
		}

		@Override
		public CompletableFuture<Void> releaseAsync(StatefulConnection<?, ?> connection) {
			
			GenericObjectPool<StatefulConnection<?, ?>> blockingPool = poolRef.remove(connection);

			if (blockingPool != null) {
				if (logger.isWarnEnabled()) {
					logger.warn("Releasing asynchronously a connection that was obtained from a non-blocking pool");
				}

				blockingPool.returnObject(connection);
				return CompletableFuture.completedFuture(null);
			}

			AsyncPool<StatefulConnection<?, ?>> pool = asyncPoolRef.remove(connection);

			if (pool == null) {
				return Futures.failed(new ConnectionPoolException("Returned connection " + connection
						+ " was either previously returned or does not belong to this connection provider"));
			}

			return pool.release(connection);
		}

		@Override
		public AbstractRedisClient getClient() {

			if (nativeConnectionProvider instanceof ClientProvider) {
				return ((ClientProvider) nativeConnectionProvider).getClient();
			}

			throw new IllegalStateException(
					String.format("Underlying connection provider %s does not implement ClientProvider.",
							nativeConnectionProvider.getClass().getName()));
		}
		
		@SuppressWarnings("rawtypes")
		@Override
		public void destroy() throws Exception {
			
			List<CompletableFuture<?>> futures = new ArrayList<>();
			
			if (!poolRef.isEmpty() || !asyncPoolRef.isEmpty()) {
				if (logger.isWarnEnabled()) {
					logger.warn("LettuceSmartConnectionPool contains unreleased connections.");
				}
			}

			if (!inProgressAsyncPoolRef.isEmpty()) {
				if (logger.isWarnEnabled()) {
					logger.warn("LettuceSmartConnectionPool has active connection retrievals.");
				}
				inProgressAsyncPoolRef.forEach((k, v) -> futures.add(k.thenApply(StatefulConnection::closeAsync)));
			}

			if (!poolRef.isEmpty()) {
				poolRef.forEach((connection, pool) -> pool.returnObject(connection));
				poolRef.clear();
			}

			if (!asyncPoolRef.isEmpty()) {
				asyncPoolRef.forEach((connection, pool) -> futures.add(pool.release(connection)));
				asyncPoolRef.clear();
			}

			pools.forEach((type, pool) -> pool.close());

			CompletableFuture
					.allOf(futures.stream().map(it -> it.exceptionally(e -> null)).toArray(CompletableFuture[]::new)) //
					.thenCompose(ignored -> {
						CompletableFuture[] poolClose = asyncPools.values().stream().map(AsyncPool::closeAsync)
								.map(it -> it.exceptionally(e -> null)).toArray(CompletableFuture[]::new);

						return CompletableFuture.allOf(poolClose);
					}) //
					.thenRun(() -> {
						asyncPoolRef.clear();
						inProgressAsyncPoolRef.clear();
					}) //
					.join();

			pools.clear();
		}

	}
	
	private static class LettucePooledObjectFactory<T extends StatefulConnection<?, ?>> extends BasePooledObjectFactory<T> {

		private final ConnectionProvider connectionProvider;
		private final Supplier<Class<T>> conditionSupplier;

		LettucePooledObjectFactory(ConnectionProvider connectionProvider, Supplier<Class<T>> conditionSupplier) {
			this.connectionProvider = connectionProvider;
			this.conditionSupplier = conditionSupplier;
		}

		@Override
		public T create() throws Exception {
			return connectionProvider.getConnection(conditionSupplier.get());
		}

		@Override
		public void destroyObject(PooledObject<T> p) throws Exception {
			p.getObject().close();
		}

		@Override
		public PooledObject<T> wrap(T obj) {
			return new DefaultPooledObject<>(obj);
		}

		@SuppressWarnings("rawtypes")
		@Override
		public void activateObject(PooledObject<T> p) throws Exception {
			T object = p.getObject();
			if (object instanceof StatefulRedisConnection && connectionProvider instanceof DatabaseProvider) {
				Integer db = Utils.getFieldValue(object, "db", Integer.class);
				int database = ((DatabaseProvider) connectionProvider).getDatabase();

				if (db != null && db.intValue() != database) {
					((StatefulRedisConnection) object).sync().select(database);
				}
			}
		}

		@Override
		public boolean validateObject(PooledObject<T> p) {
			return p.getObject().isOpen();
		}

	}

}
