/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import static org.iherus.shiro.util.Utils.assertNotNull;

import java.util.Optional;
import java.util.concurrent.CompletionStage;

import org.iherus.shiro.cache.redis.connection.lettuce.ConnectionProvider.ClientProvider;

import io.lettuce.core.AbstractRedisClient;
import io.lettuce.core.ReadFrom;
import io.lettuce.core.api.StatefulConnection;
import io.lettuce.core.cluster.RedisClusterClient;
import io.lettuce.core.cluster.api.StatefulRedisClusterConnection;
import io.lettuce.core.cluster.pubsub.StatefulRedisClusterPubSubConnection;
import io.lettuce.core.codec.RedisCodec;
import io.lettuce.core.pubsub.StatefulRedisPubSubConnection;

/**
 * LettuceClusterConnectionProvider
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class LettuceClusterConnectionProvider implements ConnectionProvider, ClientProvider {

	private final RedisClusterClient client;
	private final RedisCodec<?, ?> codec;
	private final Optional<ReadFrom> readFrom;

	private volatile boolean initialized;

	private final Object lock = new Object();

	public LettuceClusterConnectionProvider(RedisClusterClient client, RedisCodec<?, ?> codec) {
		this(client, codec, null);
	}

	public LettuceClusterConnectionProvider(RedisClusterClient client, RedisCodec<?, ?> codec, ReadFrom readFrom) {

		assertNotNull(client, "Cluster client must not be null.");
		assertNotNull(codec, "Codec must not be null.");

		this.client = client;
		this.codec = codec;
		this.readFrom = Optional.ofNullable(readFrom);
	}

	@Override
	public AbstractRedisClient getClient() {
		return client;
	}

	@Override
	public <T extends StatefulConnection<?, ?>> CompletionStage<T> getConnectionAsync(Class<T> connectionType) {

		initializePartitions();

		if (StatefulRedisClusterPubSubConnection.class.isAssignableFrom(connectionType)
				|| StatefulRedisPubSubConnection.class.equals(connectionType)) {
			return client.connectPubSubAsync(codec).thenApply(connectionType::cast);
		}

		if (StatefulRedisClusterConnection.class.isAssignableFrom(connectionType)
				|| StatefulConnection.class.equals(connectionType)) {
			return client.connectAsync(codec).thenApply(connection -> {
				readFrom.ifPresent(connection::setReadFrom);
				return connectionType.cast(connection);
			});
		}

		return Futures.failed(new UnsupportedOperationException(
				String.format("Connection type {%s} is not supported in the current provider.", connectionType)));
	}

	/**
	 * @see RedisClusterClient#connect(RedisCodec)
	 */
	protected void initializePartitions() {
		if (!initialized) {
			synchronized (lock) {
				if (!initialized) {
					client.getPartitions();
					initialized = true;
				}
			}
		}
	}

}
