/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import static io.lettuce.core.RedisURI.DEFAULT_TIMEOUT_DURATION;
import static org.iherus.shiro.util.Utils.assertNotNull;

import java.time.Duration;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.iherus.shiro.cache.redis.config.DefaultPoolConfig;
import org.iherus.shiro.cache.redis.config.HostPortPair;
import org.iherus.shiro.cache.redis.config.RedisClusterConfiguration;
import org.iherus.shiro.cache.redis.config.RedisConfiguration;
import org.iherus.shiro.cache.redis.config.RedisConfiguration.ClusterConfiguration;
import org.iherus.shiro.cache.redis.config.RedisSentinelConfiguration;
import org.iherus.shiro.cache.redis.config.RedisStandaloneConfiguration;
import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.Destroyable;
import org.iherus.shiro.cache.redis.connection.Initializable;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.cache.redis.connection.RedisConnectionFactory;
import org.iherus.shiro.cache.redis.connection.lettuce.ConnectionPool.LettuceSmartConnectionPool;
import org.iherus.shiro.exception.InvocationException;

import io.lettuce.core.AbstractRedisClient;
import io.lettuce.core.ClientOptions;
import io.lettuce.core.ReadFrom;
import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisURI;
import io.lettuce.core.cluster.ClusterClientOptions;
import io.lettuce.core.cluster.RedisClusterClient;
import io.lettuce.core.codec.ByteArrayCodec;
import io.lettuce.core.codec.RedisCodec;
import io.lettuce.core.resource.ClientResources;

/**
 * Implementation of {@link RedisConnectionFactory} a based on {@literal Lettuce}.
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class LettuceConnectionFactory implements RedisConnectionFactory, Initializable, Destroyable {

	/**
	 * @see io.lettuce.core.RedisURI#setSsl(boolean ssl)
	 */
	private boolean useSsl;

	/**
	 * @see io.lettuce.core.RedisURI#setVerifyPeer(boolean verifyPeer)
	 */
	private boolean verifyPeer = true;

	/**
	 * @see io.lettuce.core.RedisURI#setStartTls(boolean startTls)
	 */
	private boolean startTls;

	private Optional<ClientResources> clientResources = Optional.empty();
	private Optional<ClientOptions> clientOptions = Optional.empty();
	private Optional<ReadFrom> readFrom = Optional.empty();
	
	/**
	 * Default timeout: 60 sec.
	 * 
	 * Execution Policy:
	 * 
	 * <pre>
	 * long timeout = applyConnectionTimeout ? this.timeout : source.getTimeout(command);
	 * 
	 * PS: The default value of applyConnectionTimeout is false.
	 * </pre>
	 * 
	 * @see io.lettuce.core.RedisURI#setTimeout(Duration timeout)
	 * @see io.lettuce.core.cluster.api.sync.RedisClusterCommands#setTimeout(Duration
	 *      timeout)
	 * @see io.lettuce.core.TimeoutOptions#isApplyConnectionTimeout()
	 * 
	 */
	private Optional<Duration> timeout = Optional.empty();

	private Optional<String> clientName = Optional.empty();

	private RedisConfiguration configuration = RedisConfiguration.defaulted;

	@SuppressWarnings("rawtypes")
	private GenericObjectPoolConfig poolConfig;

	private volatile ConnectionPool pool;
	private volatile AbstractRedisClient client;

	private Optional<BatchOptions> batchOptions = Optional.empty();

	private final Object lock = new Object();

	public LettuceConnectionFactory() {
		this(new DefaultPoolConfig());
	}

	public LettuceConnectionFactory(GenericObjectPoolConfig<?> poolConfig) {
		this.poolConfig = poolConfig == null ? new DefaultPoolConfig() : poolConfig;
	}

	public LettuceConnectionFactory(RedisStandaloneConfiguration standaloneConfig) {
		this(standaloneConfig, new DefaultPoolConfig());
	}

	public LettuceConnectionFactory(RedisSentinelConfiguration sentinelConfig) {
		this(sentinelConfig, new DefaultPoolConfig());
	}

	public LettuceConnectionFactory(RedisClusterConfiguration clusterConfig) {
		this(clusterConfig, new DefaultPoolConfig());
	}

	public LettuceConnectionFactory(RedisStandaloneConfiguration standaloneConfig,
			GenericObjectPoolConfig<?> poolConfig) {
		this(poolConfig);
		assertNotNull(standaloneConfig, "RedisStandaloneConfiguration must not be null.");
		this.configuration = standaloneConfig;
	}

	@SuppressWarnings("rawtypes")
	public LettuceConnectionFactory(RedisSentinelConfiguration sentinelConfig, GenericObjectPoolConfig poolConfig) {
		this(poolConfig);
		assertNotNull(sentinelConfig, "RedisSentinelConfiguration must not be null.");
		this.configuration = sentinelConfig;
	}

	@SuppressWarnings("rawtypes")
	public LettuceConnectionFactory(RedisClusterConfiguration clusterConfig, GenericObjectPoolConfig poolConfig) {
		this(poolConfig);
		assertNotNull(clusterConfig, "RedisClusterConfiguration must not be null.");
		this.configuration = clusterConfig;
	}

	public boolean isUseSsl() {
		return useSsl;
	}

	public void setUseSsl(boolean useSsl) {
		this.useSsl = useSsl;
	}

	public boolean isVerifyPeer() {
		return verifyPeer;
	}

	public void setVerifyPeer(boolean verifyPeer) {
		this.verifyPeer = verifyPeer;
	}

	public boolean isStartTls() {
		return startTls;
	}

	public void setStartTls(boolean startTls) {
		this.startTls = startTls;
	}

	public Optional<ClientResources> getClientResources() {
		return clientResources;
	}

	public void setClientResources(ClientResources clientResources) {
		this.clientResources = Optional.ofNullable(clientResources);
	}

	public Optional<ClientOptions> getClientOptions() {
		return clientOptions;
	}

	public void setClientOptions(ClientOptions clientOptions) {
		this.clientOptions = Optional.ofNullable(clientOptions);
	}

	public Optional<ReadFrom> getReadFrom() {
		return readFrom;
	}

	public void setReadFrom(ReadFrom readFrom) {
		this.readFrom = Optional.ofNullable(readFrom);
	}

	public Optional<String> getClientName() {
		return clientName;
	}

	public void setClientName(String clientName) {
		this.clientName = Optional.ofNullable(clientName);
	}

	public Optional<Duration> getTimeout() {
		return timeout;
	}

	public void setTimeout(Duration timeout) {
		this.timeout = Optional.ofNullable(timeout);
	}
	
	public void setTimeoutMillis(long timeout) {
		this.setTimeout(Duration.ofMillis(timeout));
	}

	@SuppressWarnings("rawtypes")
	public GenericObjectPoolConfig getPoolConfig() {
		return poolConfig;
	}

	@SuppressWarnings("rawtypes")
	public void setPoolConfig(GenericObjectPoolConfig poolConfig) {
		this.poolConfig = poolConfig;
	}

	public RedisConfiguration getConfiguration() {
		return configuration;
	}

	public void setConfiguration(RedisConfiguration configuration) {
		this.configuration = configuration;
	}

	public Optional<BatchOptions> getBatchOptions() {
		return batchOptions;
	}

	public void setBatchOptions(BatchOptions options) {
		this.batchOptions = Optional.ofNullable(options);
	}

	protected ConnectionPool getPool() {
		if (pool == null) {
			synchronized (lock) {
				if (pool == null) {
					doInit();
				}
			}
		}
		return pool;
	}

	protected AbstractRedisClient getClient() {
		if (client == null) {
			synchronized (lock) {
				if (client == null) {
					doInit();
				}
			}
		}
		return client;
	}

	@Override
	public RedisConnection getConnection() {
		if (isRedisClusterAware()) {
			return getClusterConnection();
		}

		return new LettuceConnection(getPool(), this.batchOptions.orElse(BatchOptions.defaulted),
				this.timeout.orElse(DEFAULT_TIMEOUT_DURATION));
	}

	public LettuceClusterConnection getClusterConnection() {

		if (!isRedisClusterAware()) {
			throw new InvocationException("Cluster is not configured.");
		}

		return new LettuceClusterConnection(getPool(), this.batchOptions.orElse(BatchOptions.defaulted),
				this.timeout.orElse(DEFAULT_TIMEOUT_DURATION));
	}

	public boolean isRedisSentinelAware() {
		return RedisConfiguration.isSentinelConfiguration(configuration);
	}

	public boolean isRedisClusterAware() {
		return RedisConfiguration.isClusterConfiguration(configuration);
	}

	protected ConnectionPool createConnectionPool(AbstractRedisClient client, RedisCodec<?, ?> codec,
			ReadFrom readFrom) {
		ConnectionProvider connectionProvider = createConnectionProvider(client, codec, readFrom);
		return new LettuceSmartConnectionPool(connectionProvider, this.poolConfig);
	}

	protected ConnectionProvider createConnectionProvider(AbstractRedisClient client, RedisCodec<?, ?> codec,
			ReadFrom readFrom) {

		if (isRedisClusterAware()) {
			return new LettuceClusterConnectionProvider((RedisClusterClient) client, codec, readFrom);
		}

		return new LettuceStandaloneConnectionProvider((RedisClient) client, codec, readFrom);
	}

	protected AbstractRedisClient createClient() {

		if (isRedisSentinelAware()) {

			RedisURI redisUri = createSentinelRedisUri((RedisSentinelConfiguration) this.configuration);
			RedisClient client = getClientResources()
					.map(clientResources -> RedisClient.create(clientResources, redisUri)) //
					.orElseGet(() -> RedisClient.create(redisUri));

			getClientOptions().ifPresent(client::setOptions);
			return client;
		}

		if (isRedisClusterAware()) {

			Set<RedisURI> redisUris = createClusterRedisUris((RedisClusterConfiguration) this.configuration);

			RedisClusterClient clusterClient = getClientResources()
					.map(clientResources -> RedisClusterClient.create(clientResources, redisUris)) //
					.orElseGet(() -> RedisClusterClient.create(redisUris));

			clusterClient.setOptions(getClusterClientOptions((ClusterConfiguration) this.configuration));

			return clusterClient;
		}

		RedisStandaloneConfiguration conf = ((RedisStandaloneConfiguration) this.configuration);
		RedisURI redisUri = createRedisUri(conf.getHost(), conf.getPort(), conf.getPassword(), conf.getDatabase());

		RedisClient client = getClientResources().map(clientResources -> RedisClient.create(clientResources, redisUri)) //
				.orElseGet(() -> RedisClient.create(redisUri));
		getClientOptions().ifPresent(client::setOptions);

		return client;
	}

	private ClusterClientOptions getClusterClientOptions(ClusterConfiguration clusterConfig) {

		Optional<ClientOptions> clientOptions = getClientOptions();

		ClusterClientOptions clusterClientOptions = clientOptions //
				.filter(ClusterClientOptions.class::isInstance) //
				.map(ClusterClientOptions.class::cast) //
				.orElseGet(() -> {
					return clientOptions.map(options -> ClusterClientOptions.builder(options).build()) //
							.orElseGet(ClusterClientOptions::create);
				});

		return clusterClientOptions.mutate() //
				.maxRedirects(clusterConfig.getMaxAttempts()) //
				.build();
	}

	private RedisURI createSentinelRedisUri(RedisSentinelConfiguration sentinelConfig) {

		assertNotNull(sentinelConfig, "RedisSentinelConfiguration is required");

		Set<HostPortPair> sentinels = sentinelConfig.getSentinels();
		RedisURI.Builder builder = RedisURI.builder().withSentinelMasterId(sentinelConfig.getMasterName());

		sentinels.forEach(pair -> {
			builder.withSentinel(pair.getHost(), pair.getPort());
		});

		Optional.ofNullable(sentinelConfig.getPassword()).ifPresent(builder::withPassword);
		getClientName().ifPresent(builder::withClientName);
		getTimeout().ifPresent(builder::withTimeout);

		return builder.withSsl(isUseSsl()) //
				.withVerifyPeer(isVerifyPeer()) //
				.withStartTls(isStartTls()) //
				.withDatabase(sentinelConfig.getDatabase()) //
				.build();

	}

	private RedisURI createRedisUri(String host, int port, String password, int database) {
		RedisURI.Builder builder = RedisURI.Builder.redis(host, port);

		Optional.ofNullable(password).ifPresent(builder::withPassword);
		getClientName().ifPresent(builder::withClientName);
		getTimeout().ifPresent(builder::withTimeout);

		return builder.withSsl(isUseSsl()) //
				.withVerifyPeer(isVerifyPeer()) //
				.withStartTls(isStartTls()) //
				.withDatabase(database) //
				.build();
	}

	private Set<RedisURI> createClusterRedisUris(RedisClusterConfiguration clusterConfig) {
		assertNotNull(clusterConfig, "RedisClusterConfiguration is required");

		Set<HostPortPair> nodes = clusterConfig.getClusterNodes();

		final Set<RedisURI> redisUris = new LinkedHashSet<RedisURI>(nodes.size());

		nodes.forEach(pair -> {
			redisUris.add(createRedisUri(pair.getHost(), pair.getPort(), clusterConfig.getPassword(), 0));
		});
		return redisUris;
	}

	@Override
	public void init() throws Exception {
		doInit();
	}

	private void doInit() {
		if (this.client == null) {
			this.client = createClient();
			this.pool = createConnectionPool(this.client, ByteArrayCodec.INSTANCE, this.readFrom.orElse(null));
		}
	}

	@Override
	public void destroy() throws Exception {
		doDestroy();
	}

	private void doDestroy() throws Exception {
		if (this.client != null) {
			try {
				this.client.shutdown();
				this.pool.destroy();
			} finally {
				this.client = null;
				this.pool = null;
			}
		}
	}

}
