/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.lettuce;

import io.lettuce.core.AbstractRedisClient;
import io.lettuce.core.ReadFrom;
import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisURI;
import io.lettuce.core.api.StatefulConnection;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.cluster.pubsub.StatefulRedisClusterPubSubConnection;
import io.lettuce.core.codec.RedisCodec;
import io.lettuce.core.masterreplica.MasterReplica;
import io.lettuce.core.masterreplica.StatefulRedisMasterReplicaConnection;
import io.lettuce.core.pubsub.StatefulRedisPubSubConnection;
import io.lettuce.core.sentinel.api.StatefulRedisSentinelConnection;
import org.iherus.shiro.cache.redis.connection.lettuce.ConnectionProvider.ClientProvider;
import org.iherus.shiro.cache.redis.connection.lettuce.StandaloneConnectionProvider.DatabaseProvider;
import org.iherus.shiro.util.Utils;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

import static org.iherus.shiro.util.Utils.assertNotNull;

/**
 * LettuceStandaloneConnectionProvider
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class LettuceStandaloneConnectionProvider implements StandaloneConnectionProvider, ClientProvider, DatabaseProvider {

	private final RedisClient client;
	private final RedisCodec<?, ?> codec;
	private final Optional<ReadFrom> readFrom;
	private final RedisURI redisUri;

	public LettuceStandaloneConnectionProvider(RedisClient client, RedisCodec<?, ?> codec) {
		this(client, codec, null);
	}

	public LettuceStandaloneConnectionProvider(RedisClient client, RedisCodec<?, ?> codec, ReadFrom readFrom) {
		assertNotNull(client, "Client must not be null.");
		assertNotNull(codec, "Codec must not be null.");

		this.client = client;
		this.codec = codec;
		this.readFrom = Optional.ofNullable(readFrom);
		this.redisUri = Utils.getFieldValue(this.client, "redisURI", RedisURI.class);
	}

	@Override
	public AbstractRedisClient getClient() {
		return client;
	}

	@Override
	public int getDatabase() {
		return this.redisUri.getDatabase();
	}

	@Override
	public <T extends StatefulConnection<?, ?>> CompletionStage<T> getConnectionAsync(Class<T> connectionType) {
		return getConnectionAsync(connectionType, this.redisUri);
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public <T extends StatefulConnection<?, ?>> CompletionStage<T> getConnectionAsync(Class<T> connectionType,
			RedisURI redisUri) {

		if (StatefulRedisSentinelConnection.class.isAssignableFrom(connectionType)) {
			return client.connectSentinelAsync(codec, redisUri).thenApply(connectionType::cast);
		}

		if (StatefulRedisPubSubConnection.class.isAssignableFrom(connectionType)
				&& (!StatefulRedisClusterPubSubConnection.class.isAssignableFrom(connectionType))) {

			return client.connectPubSubAsync(codec, redisUri).thenApply(connectionType::cast);
		}

		if (StatefulRedisConnection.class.isAssignableFrom(connectionType)
				|| StatefulConnection.class.equals(connectionType)) {

			return readFrom.map(rf -> this.masterReplicaConnectionAsync(redisUri, rf)) //
					.orElseGet(() -> (CompletionStage) client.connectAsync(codec, redisUri)) //
					.thenApply(connectionType::cast);
		}

		return Futures.failed(new UnsupportedOperationException(
				String.format("Connection type {%s} is not supported in the current provider.", connectionType)));
	}

	private CompletionStage<StatefulRedisConnection<?, ?>> masterReplicaConnectionAsync(RedisURI redisUri,
																						ReadFrom readFrom) {
		CompletableFuture<? extends StatefulRedisMasterReplicaConnection<?, ?>> connection = MasterReplica.connectAsync(client, codec, redisUri);
		return connection.thenApply(conn -> {
			conn.setReadFrom(readFrom);
			return conn;
		});
	}

}
