/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.redisson;

import static org.iherus.shiro.util.Utils.bytesToText;
import static org.iherus.shiro.util.Utils.intToBytes;
import static org.iherus.shiro.util.Utils.newMutableArray;
import static org.redisson.client.codec.ByteArrayCodec.INSTANCE;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;

import org.iherus.shiro.cache.redis.Constant;
import org.iherus.shiro.cache.redis.connection.AbstractRedisConnection;
import org.iherus.shiro.util.Utils.MutableArray;
import org.redisson.ScanResult;
import org.redisson.api.RFuture;
import org.redisson.client.RedisClient;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.client.protocol.decoder.ListScanResult;
import org.redisson.command.CommandExecutor;
import org.redisson.connection.MasterSlaveEntry;

/**
 * AbstractRedissonConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public abstract class AbstractRedissonConnection extends AbstractRedisConnection {

	protected abstract CommandExecutor getCommandExecutor();

	public Iterator<byte[]> scanKeys(MasterSlaveEntry entry, final byte[] pattern, final int batchSize) {

		return new ByteArrayIterator() {

			@Override
			protected ScanResult<byte[]> iterator(RedisClient client, long nextIterPos) {
				return getCommandExecutor().get(scanAsync(client, entry, nextIterPos, pattern, batchSize));
			}
		};

	}

	private RFuture<ListScanResult<byte[]>> scanAsync(RedisClient client, MasterSlaveEntry entry, long startPos,
			byte[] pattern, int count) {
		return getCommandExecutor().readAsync(client, entry, INSTANCE, RedisCommands.SCAN, startPos, "MATCH",
				((Object) pattern), "COUNT", count);
	}
	
	@SuppressWarnings("unchecked")
	public Set<byte[]> scanKeysOnDb(MasterSlaveEntry entry, final int db, final byte[] pattern,
			final int batchSize) {
		Set<byte[]> keys = new HashSet<byte[]>();

		final MutableArray<Object> mutableArray = newMutableArray(null, (Object) pattern,
				(Object) intToBytes(batchSize), (Object) intToBytes(db));

		byte[] cursor = intToBytes(0);

		do {
			Object[] params = mutableArray.replace(0, (Object) cursor).toArray();

			RFuture<Object> f = getCommandExecutor().evalReadAsync(entry, INSTANCE, RedisCommands.EVAL_LIST,
					Constant.Select.SCAN.command(), Collections.emptyList(), params);

			List<Object> objectList = (List<Object>) getCommandExecutor().get(f);

			cursor = (byte[]) objectList.get(0);
			keys.addAll((Collection<? extends byte[]>) objectList.get(1));

		} while (bytesToText(cursor).compareTo("0") > 0);

		return keys;
	}
	

	/**
	 * Created to enhance compatibility.
	 */
	static abstract class ByteArrayIterator implements Iterator<byte[]> {

		private Iterator<byte[]> lastIter;
		protected long nextIterPos;
		protected RedisClient client;

		private boolean finished;
		private boolean currentElementRemoved;
		protected byte[] value;

		@Override
		public boolean hasNext() {
			if (lastIter == null || !lastIter.hasNext()) {
				if (finished) {
					currentElementRemoved = false;
					client = null;
					nextIterPos = 0;

					if (!tryAgain()) {
						return false;
					}
					finished = false;
				}
				do {
					ScanResult<byte[]> res = iterator(client, nextIterPos);

					client = res.getRedisClient();

					lastIter = res.getValues().iterator();
					nextIterPos = res.getPos();

					if (res.getPos() == 0) {
						finished = true;
						if (res.getValues().isEmpty()) {
							currentElementRemoved = false;

							client = null;
							nextIterPos = 0;
							if (tryAgain()) {
								continue;
							}

							return false;
						}
					}
				} while (!lastIter.hasNext());
			}
			return lastIter.hasNext();
		}

		protected boolean tryAgain() {
			return false;
		}

		protected abstract ScanResult<byte[]> iterator(RedisClient client, long nextIterPos);

		@Override
		public byte[] next() {
			if (!hasNext()) {
				throw new NoSuchElementException("No such element");
			}

			value = lastIter.next();
			currentElementRemoved = false;
			return value;
		}

		@Override
		public void remove() {
			if (currentElementRemoved) {
				throw new IllegalStateException("Element been already deleted");
			}
			if (lastIter == null || value == null) {
				throw new IllegalStateException();
			}

			lastIter.remove();
			currentElementRemoved = true;
		}

	}

}
