/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.redisson;

import static org.iherus.shiro.cache.redis.Constant.GETDEL;
import static org.iherus.shiro.cache.redis.Constant.GETSET;
import static org.iherus.shiro.util.Utils.longToBytes;
import static org.redisson.client.codec.ByteArrayCodec.INSTANCE;
import static org.redisson.client.protocol.RedisCommands.DEL;
import static org.redisson.client.protocol.RedisCommands.EVAL_OBJECT;
import static org.redisson.client.protocol.RedisCommands.GET;
import static org.redisson.client.protocol.RedisCommands.INFO_SERVER;
import static org.redisson.client.protocol.RedisCommands.MGET;
import static org.redisson.client.protocol.RedisCommands.UNLINK;
import static org.redisson.connection.MasterSlaveConnectionManager.MAX_SLOT;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;

import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.util.RedisVerUtils;
import org.iherus.shiro.util.Utils;
import org.redisson.Redisson;
import org.redisson.api.RFuture;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.StringCodec;
import org.redisson.command.CommandExecutor;
import org.redisson.connection.CRC16;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;

/**
 * RedissonClusterConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class RedissonClusterConnection extends AbstractRedissonConnection implements RedisConnection {

	private final Redisson redisson;
	private final BatchOptions options;

	private static final Function<byte[], Integer> calculator = ((key) -> {
		return RedissonClusterConnection.calcSlot(key);
	});

	public RedissonClusterConnection(RedissonClient client) {
		this(client, BatchOptions.defaulted);
	}

	public RedissonClusterConnection(RedissonClient client, BatchOptions options) {
		this.redisson = (Redisson) client;
		this.options = options;
	}

	@Override
	protected CommandExecutor getCommandExecutor() {
		return this.redisson.getCommandExecutor();
	}

	@Override
	public byte[] get(byte[] key) {
		RFuture<byte[]> f = getCommandExecutor().readAsync(key, INSTANCE, GET, (Object) key);
		return getCommandExecutor().get(f);
	}

	@Override
	public byte[] set(byte[] key, byte[] value, Duration expired) {
		MasterSlaveEntry entry = getEntryByKey(key);

		RFuture<byte[]> f = getCommandExecutor().evalWriteAsync(entry, INSTANCE, EVAL_OBJECT, GETSET,
				Collections.singletonList((Object) key), (Object) value, (Object) longToBytes(Duration.ZERO.equals(expired) ? -1l : expired.toMillis()));
		return getCommandExecutor().get(f);
	}

	@Override
	public Long mdel(byte[]... keys) {
		if (Utils.isEmpty(keys)) return 0L;

		final CommandExecutor executor = getCommandExecutor();

		boolean unlink = RedisVerUtils.getServerVersion(() -> {
			MasterSlaveEntry entry = executor.getConnectionManager().getEntrySet().iterator().next();
			RFuture<Map<String, String>> f = executor.readAsync(entry, StringCodec.INSTANCE, INFO_SERVER);
			return executor.get(f).getOrDefault("redis_version", "");
		}).isSupportUnlink();

		return batchDeleteOnCluster(this.options.getDeleteBatchSize(), keys, ((batchKeys) -> {
			RFuture<Long> f = executor.writeAsync(batchKeys[0], INSTANCE, unlink ? UNLINK : DEL,
					Arrays.asList(batchKeys).toArray());
			return executor.get(f);
		}), calculator);
	}

	@Override
	public List<byte[]> mget(byte[]... keys) {
		final CommandExecutor executor = getCommandExecutor();

		return batchGetOnCluster(this.options.getFetchBatchSize(), keys, (batchKeys) -> {
			RFuture<List<byte[]>> f = executor.readAsync(batchKeys[0], INSTANCE, MGET,
					Arrays.asList(batchKeys).toArray());
			return executor.get(f);
		}, calculator);
	}

	@Override
	public byte[] del(byte[] key) {
		MasterSlaveEntry entry = getEntryByKey(key);
		RFuture<byte[]> f = getCommandExecutor().evalWriteAsync(entry, INSTANCE, EVAL_OBJECT, GETDEL,
				Collections.singletonList((Object) key));
		return getCommandExecutor().get(f);
	}

	@Override
	public Set<byte[]> keys(byte[] pattern) {
		return distributionScanKeys((completion) -> {
			Collection<MasterSlaveEntry> entrySet = getCommandExecutor().getConnectionManager().getEntrySet();
			entrySet.forEach(entry -> {
				completion.submit(() -> {
					Set<byte[]> keysOfNode = new HashSet<>();
					Iterator<byte[]> iterator = scanKeys(entry, pattern, this.options.getScanBatchSize());
					while (iterator.hasNext()) {
						keysOfNode.add(iterator.next());
					}
					return keysOfNode;
				});
			});
			return entrySet.size();
		});
	}

	@Override
	protected ExecutorService getExecutor() {
		return this.redisson.getConnectionManager().getExecutor();
	}

	@Override
	public boolean isClusterConnection() {
		return true;
	}

	private MasterSlaveEntry getEntryByKey(byte[] key) {
		ConnectionManager cm = this.redisson.getConnectionManager();
		return cm.getEntry(cm.calcSlot(key));
	}

	private static int calcSlot(byte[] key) {
		if (key == null) {
			return 0;
		}

		int start = indexOf(key, (byte) '{');
		if (start != -1) {
			int end = indexOf(key, (byte) '}');
			key = Arrays.copyOfRange(key, start + 1, end);
		}

		int result = CRC16.crc16(key) % MAX_SLOT;
		return result;
	}

	private static int indexOf(byte[] array, byte element) {
		for (int i = 0; i < array.length; ++i) {
			if (array[i] == element) {
				return i;
			}
		}
		return -1;
	}

	@Override
	public void close() {
		// do nothing
	}

}
