/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.redisson;

import static org.iherus.shiro.cache.redis.Constant.GETDEL;
import static org.iherus.shiro.cache.redis.Constant.GETSET;
import static org.iherus.shiro.util.Utils.intToBytes;
import static org.iherus.shiro.util.Utils.longToBytes;
import static org.iherus.shiro.util.Utils.newMutableArray;
import static org.redisson.client.codec.ByteArrayCodec.INSTANCE;
import static org.redisson.client.protocol.RedisCommands.DEL;
import static org.redisson.client.protocol.RedisCommands.GET;
import static org.redisson.client.protocol.RedisCommands.INFO_SERVER;
import static org.redisson.client.protocol.RedisCommands.MGET;
import static org.redisson.client.protocol.RedisCommands.UNLINK;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.iherus.shiro.cache.redis.Constant;
import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.MutableDatabase;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.util.RedisVerUtils;
import org.iherus.shiro.util.Utils;
import org.redisson.Redisson;
import org.redisson.api.RFuture;
import org.redisson.api.RScript;
import org.redisson.api.RScript.Mode;
import org.redisson.api.RScript.ReturnType;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.StringCodec;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.command.CommandExecutor;
import org.redisson.connection.MasterSlaveEntry;

/**
 * RedissonConnection
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class RedissonConnection extends AbstractRedissonConnection implements RedisConnection, MutableDatabase {

	private final Redisson redisson;
	private final BatchOptions options;

	private Optional<Integer> database = Optional.empty();

	public RedissonConnection(RedissonClient client) {
		this(client, BatchOptions.defaulted);
	}

	public RedissonConnection(RedissonClient client, BatchOptions options) {
		this.redisson = (Redisson) client;
		this.options = options;
	}

	public Optional<Integer> getDatabase() {
		return database;
	}

	@Override
	public void setDatabase(Integer database) {
		this.database = Optional.ofNullable(database == null ? null : Math.max(0, database));
	}

	protected boolean isSwitchDb() {
		int nativeDb = this.redisson.getConnectionManager().getConfig().getDatabase();
		return getDatabase().filter(db -> db.intValue() != nativeDb).isPresent();
	}

	public Redisson getRedisson() {
		return redisson;
	}

	public BatchOptions getOptions() {
		return options;
	}

	@Override
	protected CommandExecutor getCommandExecutor() {
		return this.redisson.getCommandExecutor();
	}

	@Override
	public byte[] get(byte[] key) {
		boolean isSwitchDb = isSwitchDb();

		RFuture<byte[]> f = null;
		
		if (isSwitchDb) {
			f = getCommandExecutor().evalReadAsync("", INSTANCE, RedisCommands.EVAL_OBJECT,
					Constant.Select.GET.command(), Collections.singletonList((Object) key),
					(Object) intToBytes(getDatabase().get()));
		} else {
			f = getCommandExecutor().readAsync(key, INSTANCE, GET, (Object) key);
		}
		
		return getCommandExecutor().get(f);
	}

	@Override
	public byte[] set(byte[] key, byte[] value, Duration expired) {
		boolean isSwitchDb = isSwitchDb();

		RScript script = this.redisson.getScript(INSTANCE);

		Object[] params = newMutableArray((Object) value, (Object) longToBytes(Duration.ZERO.equals(expired) ? -1l : expired.toMillis()))
				.add(isSwitchDb, () -> intToBytes(getDatabase().get())).toArray();

		RFuture<byte[]> f = script.evalAsync(Mode.READ_WRITE, isSwitchDb ? Constant.Select.GETSET.command() : GETSET,
				ReturnType.VALUE, Collections.singletonList(key), params);

		return getCommandExecutor().get(f);
	}

	@Override
	public Long mdel(byte[]... keys) {
		if (Utils.isEmpty(keys)) return 0L;
		
		final CommandExecutor executor = getCommandExecutor();

		boolean unlink = RedisVerUtils.getServerVersion(() -> {
			Map<String, String> server = executor.read(null, StringCodec.INSTANCE, INFO_SERVER);
			return server.getOrDefault("redis_version", "");
		}).isSupportUnlink();

		boolean isSwitchDb = isSwitchDb();

		return batchDeleteOnStandalone(options.getDeleteBatchSize(), keys, (batchKeys) -> {

			RFuture<Long> f = null;

			if (isSwitchDb) {
				f = executor.evalWriteAsync("", INSTANCE, RedisCommands.EVAL_LONG,
						(unlink ? Constant.Select.UNLINK : Constant.Select.DEL).command(), wrapKeys(batchKeys),
						(Object) intToBytes(getDatabase().get()));

			} else {
				f = executor.writeAsync((String) null, INSTANCE, unlink ? UNLINK : DEL,
						Arrays.asList(batchKeys).toArray());
			}

			return executor.get(f);
		});
	}

	@Override
	public List<byte[]> mget(byte[]... keys) {
		boolean isSwitchDb = isSwitchDb();

		final CommandExecutor executor = getCommandExecutor();
		return batchGetOnStandalone(this.options.getFetchBatchSize(), keys, (batchKeys) -> {
			RFuture<List<byte[]>> f = null;

			if (isSwitchDb) {
				f = executor.evalReadAsync("", INSTANCE, RedisCommands.EVAL_LIST, Constant.Select.MGET.command(),
						wrapKeys(batchKeys), (Object) intToBytes(getDatabase().get()));

			} else {
				f = executor.readAsync("", INSTANCE, MGET, Arrays.asList(batchKeys).toArray());
			}

			return executor.get(f);
		});
	}

	private static List<Object> wrapKeys(byte[]... keys) {
		return Collections.unmodifiableList(new ArrayList<Object>(Arrays.asList(keys)));
	}

	@Override
	public byte[] del(byte[] key) {
		boolean isSwitchDb = isSwitchDb();

		RScript script = this.redisson.getScript(INSTANCE);

		RFuture<Object> f = script.evalAsync(Mode.READ_WRITE, isSwitchDb ? Constant.Select.GETDEL.command() : GETDEL,
				ReturnType.VALUE, Collections.singletonList(key),
				isSwitchDb ? new Object[] { (Object) intToBytes(getDatabase().get()) } : new Object[] {});

		return (byte[]) getCommandExecutor().get(f);
	}

	@Override
	public Set<byte[]> keys(byte[] pattern) {
		boolean isSwitchDb = isSwitchDb();

		Set<byte[]> keys = new HashSet<byte[]>();
		for (MasterSlaveEntry entry : getCommandExecutor().getConnectionManager().getEntrySet()) {
			if (isSwitchDb) {
				keys.addAll(scanKeysOnDb(entry, getDatabase().get(), pattern, this.options.getScanBatchSize()));
				continue;
			}
			Iterator<byte[]> iterator = scanKeys(entry, pattern, this.options.getScanBatchSize());
			while (iterator.hasNext()) {
				keys.add(iterator.next());
			}
		}
		return keys;
	}

	@Override
	public void close() {
		// do nothing
	}

}
