/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.connection.spring;

import org.iherus.shiro.cache.redis.config.RedisConfiguration;
import org.iherus.shiro.cache.redis.connection.BatchOptions;
import org.iherus.shiro.cache.redis.connection.RedisConnection;
import org.iherus.shiro.cache.redis.connection.RedisConnectionFactory;
import org.iherus.shiro.cache.redis.connection.redisson.RedissonClusterConnection;
import org.iherus.shiro.cache.redis.connection.redisson.RedissonConnection;
import org.redisson.api.RedissonClient;
import org.springframework.data.redis.connection.RedisClusterConnection;
import org.springframework.util.ClassUtils;

import java.util.Optional;

import static org.iherus.shiro.util.Utils.assertNotNull;

/**
 * Implemention for compatibility with
 * {@link org.springframework.data.redis.connection.RedisConnectionFactory}
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class CompatibleRedisConnectionFactory implements RedisConnectionFactory {

	private final org.springframework.data.redis.connection.RedisConnectionFactory nativeConnectionFactory;

	private Optional<BatchOptions> batchOptions = Optional.empty();

	private static boolean importRedissonComponent;

	static {
		importRedissonComponent = ClassUtils.isPresent("org.redisson.spring.data.connection.RedissonConnection", null);
	}

	public CompatibleRedisConnectionFactory(
			org.springframework.data.redis.connection.RedisConnectionFactory connectionFactory) {
		assertNotNull(connectionFactory, "org.springframework.data.redis.connection.RedisConnectionFactory cannot be null.");
		this.nativeConnectionFactory = connectionFactory;
	}

	public Optional<BatchOptions> getBatchOptions() {
		return batchOptions;
	}

	public void setBatchOptions(BatchOptions batchOptions) {
		this.batchOptions = Optional.ofNullable(batchOptions);
	}

	public org.springframework.data.redis.connection.RedisConnectionFactory getNativeConnectionFactory() {
		return nativeConnectionFactory;
	}

	@Override
	public RedisConnection getConnection() {
		BatchOptions batchOptionsToUse = this.batchOptions.orElse(BatchOptions.defaulted);
		org.springframework.data.redis.connection.RedisConnection connection = this.nativeConnectionFactory.getConnection();
		if (importRedissonComponent) {
			return ((connection instanceof RedisClusterConnection) ?
					new RedissonClusterConnection((RedissonClient) connection.getNativeConnection(), batchOptionsToUse) :
					new RedissonConnection((RedissonClient) connection.getNativeConnection(), batchOptionsToUse));
		}
		return new CompatibleRedisConnection(connection, batchOptionsToUse);
	}

	@Override
	public RedisConfiguration getConfiguration() {
		throw new UnsupportedOperationException("This operation is not supported.");
	}

}