/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.serializer;

import static org.iherus.shiro.util.Utils.isEmpty;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import org.apache.shiro.io.ClassResolvingObjectInputStream;
import org.apache.shiro.io.SerializationException;

/**
 * <p>序列化工具类</p>
 * <p>Description:基于JDK原生实现，支持序列化和反序列化.</p>
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class DefaultSerializer extends ValueSerializer {

	public static final DefaultSerializer INSTANCE = new DefaultSerializer();

	@Override
	public byte[] serialize(Object object) throws SerializationException {
		if (object == null) {
			return EMPTY_ARRAY;
		}
		
		if (!(object instanceof Serializable)) {
			throw new IllegalArgumentException(getClass().getSimpleName() + " requires a Serializable payload "
					+ "but received an object of type [" + object.getClass().getName() + "]");
		}
		
		ByteArrayOutputStream bytesOutputStream = new ByteArrayOutputStream(DEFAULT_BUFFER_SIZE);
		try {
			ObjectOutputStream objectOutputStream = new ObjectOutputStream(bytesOutputStream);
			objectOutputStream.writeObject(object);
			objectOutputStream.flush();
			return bytesOutputStream.toByteArray();
		} catch (Exception e) {
			throw new SerializationException("Failed to serialize object of type: " + object.getClass().getName(), e);
		}
	}

	@Override
	public Object deserialize(byte[] serialized) throws SerializationException {
		if (isEmpty(serialized)) {
			return null;
		}
		ByteArrayInputStream bytesInputStream = new ByteArrayInputStream(serialized);
		ObjectInputStream objectInputStream = null;
		try {
			objectInputStream = new ClassResolvingObjectInputStream(bytesInputStream);
			return objectInputStream.readObject();
		} catch (Exception e) {
			throw new SerializationException("Failed to deserialize payload. " 
					+ "Is the byte array a result of corresponding serialization for "
					+ this.getClass().getSimpleName() + "?", e);
		} finally {
			if (objectInputStream != null) {
				try {
					objectInputStream.close();
				} catch (IOException e) {
					// ignore
				}
			}
		}
	}

}