/**
 * Copyright (c) 2016-2021, Bosco.Liao (bosco_liao@126.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.iherus.shiro.cache.redis.serializer;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import org.apache.shiro.io.SerializationException;
import org.apache.shiro.io.Serializer;
import org.iherus.shiro.util.Utils;


/**
 * <p>字符串序列化器</p>
 * <p>Description:基于字符串与字节数组的序列化和反序列化</p>
 * 
 * @author Bosco.Liao
 * @since 2.0.0
 */
public class StringSerializer implements Serializer<String> {

	public static final StringSerializer UTF_8 = new StringSerializer();

	private final Charset charset;

	public StringSerializer() {
		this(StandardCharsets.UTF_8);
	}

	public StringSerializer(Charset charset) {
		Utils.assertNotNull(charset, "Charset must not be null!");
		this.charset = charset;
	}

	@Override
	public byte[] serialize(String string) throws SerializationException {
		return (string == null ? null : string.getBytes(charset));
	}

	@Override
	public String deserialize(byte[] bytes) throws SerializationException {
		return (bytes == null ? null : new String(bytes, charset));
	}

}