/*
 * Decompiled with CFR 0.152.
 */
package org.imixs.archive.service.export;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.logging.Logger;
import javax.annotation.Resource;
import javax.ejb.ScheduleExpression;
import javax.ejb.Stateless;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerConfig;
import javax.ejb.TimerService;
import javax.inject.Inject;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.imixs.archive.service.ArchiveException;
import org.imixs.archive.service.MessageService;
import org.imixs.archive.service.cassandra.ClusterService;
import org.imixs.archive.service.cassandra.DataService;
import org.imixs.archive.service.export.FTPConnector;
import org.imixs.workflow.ItemCollection;
import org.imixs.workflow.xml.XMLDocumentAdapter;

@Stateless
public class ExportService {
    public static final String TIMER_ID_EXPORTSERVICE = "IMIXS_ARCHIVE_EXPORT_TIMER";
    public static final String ITEM_EXPORTPOINT = "export.point";
    public static final String ITEM_EXPORTCOUNT = "export.count";
    public static final String ITEM_EXPORTSIZE = "export.size";
    public static final String ITEM_EXPORTERRORS = "export.errors";
    public static final String DEFAULT_SCHEDULER_DEFINITION = "hour=*";
    public static final String ENV_EXPORT_SCHEDULER_DEFINITION = "EXPORT_SCHEDULER_DEFINITION";
    public static final String ENV_EXPORT_FTP_HOST = "EXPORT_FTP_HOST";
    public static final String ENV_EXPORT_FTP_PATH = "EXPORT_FTP_PATH";
    public static final String ENV_EXPORT_FTP_PORT = "EXPORT_FTP_PORT";
    public static final String ENV_EXPORT_FTP_USER = "EXPORT_FTP_USER";
    public static final String ENV_EXPORT_FTP_PASSWORD = "EXPORT_FTP_PASSWORD";
    public static final String MESSAGE_TOPIC = "export";
    @Inject
    @ConfigProperty(name="EXPORT_SCHEDULER_DEFINITION", defaultValue="")
    String schedulerDefinition;
    @Inject
    @ConfigProperty(name="EXPORT_FTP_HOST", defaultValue="")
    String ftpServer;
    @Resource
    TimerService timerService;
    @Inject
    DataService dataService;
    @Inject
    ClusterService clusterService;
    @Inject
    MessageService messageService;
    @Inject
    FTPConnector ftpConnector;
    private static Logger logger = Logger.getLogger(ExportService.class.getName());

    public boolean startScheduler() throws ArchiveException {
        try {
            logger.info("...starting the export scheduler...");
            if (this.clusterService.getSession() != null) {
                this.start();
                return true;
            }
            logger.warning("...Failed to initalize imixs-archive keyspace!");
            return false;
        }
        catch (Exception e) {
            logger.warning("...Failed to initalize imixsarchive keyspace: " + e.getMessage());
            return false;
        }
    }

    public boolean stopScheduler() throws ArchiveException {
        this.stop(this.findTimer());
        return true;
    }

    public Date getNextTimeout() {
        try {
            Timer timer = this.findTimer();
            if (timer != null) {
                return timer.getNextTimeout();
            }
        }
        catch (Exception e) {
            logger.warning("unable to updateTimerDetails: " + e.getMessage());
        }
        return null;
    }

    private void start() throws ArchiveException {
        Timer timer = null;
        timer = this.findTimer();
        if (timer != null) {
            try {
                timer.cancel();
                timer = null;
            }
            catch (Exception e) {
                this.messageService.logMessage(MESSAGE_TOPIC, "Failed to stop existing timer - " + e.getMessage());
                throw new ArchiveException(ExportService.class.getName(), "INVALID_WORKITEM", " failed to cancel existing timer!");
            }
        }
        try {
            logger.finest("...starting scheduler export-service ...");
            timer = this.createTimerOnCalendar();
            if (timer != null) {
                this.messageService.logMessage(MESSAGE_TOPIC, "Timer started.");
            }
        }
        catch (ParseException e) {
            throw new ArchiveException(ExportService.class.getName(), "INVALID_WORKITEM", " failed to start timer: " + e.getMessage());
        }
    }

    private void stop(Timer timer) throws ArchiveException {
        if (timer != null) {
            try {
                timer.cancel();
            }
            catch (Exception e) {
                this.messageService.logMessage(MESSAGE_TOPIC, "Failed to stop timer - " + e.getMessage());
            }
            this.messageService.logMessage(MESSAGE_TOPIC, "Timer stopped. ");
        }
    }

    private Timer findTimer() {
        for (Object obj : this.timerService.getTimers()) {
            Timer timer = (Timer)obj;
            if (!TIMER_ID_EXPORTSERVICE.equals(timer.getInfo())) continue;
            return timer;
        }
        return null;
    }

    Timer createTimerOnCalendar() throws ParseException, ArchiveException {
        String[] calendarConfiguation;
        TimerConfig timerConfig = new TimerConfig();
        timerConfig.setInfo((Serializable)((Object)TIMER_ID_EXPORTSERVICE));
        ScheduleExpression scheduerExpression = new ScheduleExpression();
        if (this.schedulerDefinition.isEmpty()) {
            this.messageService.logMessage(MESSAGE_TOPIC, "no scheduler definition found!");
            return null;
        }
        for (String confgEntry : calendarConfiguation = this.schedulerDefinition.split("(\\r?\\n)|(;)|(,)")) {
            Date convertedDate;
            SimpleDateFormat dateFormat;
            if (confgEntry.startsWith("second=")) {
                scheduerExpression.second(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("minute=")) {
                scheduerExpression.minute(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("hour=")) {
                scheduerExpression.hour(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("dayOfWeek=")) {
                scheduerExpression.dayOfWeek(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("dayOfMonth=")) {
                scheduerExpression.dayOfMonth(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("month=")) {
                scheduerExpression.month(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("year=")) {
                scheduerExpression.year(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("timezone=")) {
                scheduerExpression.timezone(confgEntry.substring(confgEntry.indexOf(61) + 1));
            }
            if (confgEntry.startsWith("start=")) {
                dateFormat = new SimpleDateFormat("yyyy/MM/dd");
                convertedDate = dateFormat.parse(confgEntry.substring(confgEntry.indexOf(61) + 1));
                scheduerExpression.start(convertedDate);
            }
            if (!confgEntry.startsWith("end=")) continue;
            dateFormat = new SimpleDateFormat("yyyy/MM/dd");
            convertedDate = dateFormat.parse(confgEntry.substring(confgEntry.indexOf(61) + 1));
            scheduerExpression.end(convertedDate);
        }
        Timer timer = this.timerService.createCalendarTimer(scheduerExpression, timerConfig);
        return timer;
    }

    @Timeout
    void onTimeout(Timer timer) throws Exception {
        long syncPoint = 0L;
        int exportCount = 0;
        long exportSize = 0L;
        long exportErrors = 0L;
        long totalCount = 0L;
        long totalSize = 0L;
        long latestExportPoint = 0L;
        ItemCollection metaData = null;
        String lastUniqueID = null;
        if (this.ftpServer.isEmpty()) {
            this.messageService.logMessage(MESSAGE_TOPIC, "...Export failed - EXPORT_FTP_HOST not defined!");
            this.stop(timer);
            return;
        }
        try {
            metaData = this.dataService.loadMetadata();
            syncPoint = metaData.getItemValueLong(ITEM_EXPORTPOINT);
            totalCount = metaData.getItemValueLong(ITEM_EXPORTCOUNT);
            totalSize = metaData.getItemValueLong(ITEM_EXPORTSIZE);
            latestExportPoint = syncPoint;
            this.messageService.logMessage(MESSAGE_TOPIC, "...Export started, syncpoint=" + new Date(syncPoint) + "...");
            Date now = new Date();
            if (syncPoint > now.getTime()) {
                logger.warning("...current syncpoint (" + syncPoint + ") is in the future! Adjust Syncpoint to now (" + now.getTime() + ")....");
                syncPoint = now.getTime();
            }
            LocalDate localDateSyncPoint = new Date(syncPoint).toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            LocalDate localDateNow = new Date().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            while (!localDateSyncPoint.isAfter(localDateNow)) {
                int localCount = 0;
                long lProfiler = System.currentTimeMillis();
                logger.info("...export sync point=" + localDateSyncPoint);
                List result = this.dataService.loadSnapshotsByDate(localDateSyncPoint);
                for (String snapshotID : result) {
                    ItemCollection snapshot;
                    Date snaptshotDate;
                    String latestSnapshot = this.findLatestSnapshotID(snapshotID);
                    if (latestSnapshot.equals(snapshotID) && (snaptshotDate = (snapshot = this.dataService.loadSnapshot(snapshotID)).getItemValueDate("$modified")).getTime() > syncPoint) {
                        this.ftpConnector.put(snapshot);
                        long _tmpSize = this.dataService.calculateSize(XMLDocumentAdapter.getDocument((ItemCollection)snapshot));
                        logger.finest("......size=: " + _tmpSize);
                        exportSize += _tmpSize;
                        ++exportCount;
                        ++localCount;
                        if (snaptshotDate.getTime() > latestExportPoint) {
                            latestExportPoint = snaptshotDate.getTime();
                        }
                    }
                    if (localCount < 100) continue;
                    this.messageService.logMessage(MESSAGE_TOPIC, "...... [" + localDateSyncPoint + "] " + localCount + " snapshots exported in " + (System.currentTimeMillis() - lProfiler) + "ms, last export=" + new Date(latestExportPoint));
                    localCount = 0;
                    metaData.setItemValue(ITEM_EXPORTPOINT, (Object)latestExportPoint);
                    metaData.setItemValue(ITEM_EXPORTCOUNT, (Object)(totalCount + (long)exportCount));
                    metaData.setItemValue(ITEM_EXPORTSIZE, (Object)(totalSize + exportSize));
                    metaData.setItemValue(ITEM_EXPORTERRORS, (Object)exportErrors);
                    this.dataService.saveMetadata(metaData);
                }
                metaData.setItemValue(ITEM_EXPORTPOINT, (Object)latestExportPoint);
                metaData.setItemValue(ITEM_EXPORTCOUNT, (Object)(totalCount + (long)exportCount));
                metaData.setItemValue(ITEM_EXPORTSIZE, (Object)(totalSize + exportSize));
                metaData.setItemValue(ITEM_EXPORTERRORS, (Object)exportErrors);
                this.dataService.saveMetadata(metaData);
                localDateSyncPoint = localDateSyncPoint.plusDays(1L);
            }
            this.messageService.logMessage(MESSAGE_TOPIC, "...Export finished, " + exportCount + " snapshots exported in total = " + this.messageService.userFriendlyBytes(exportSize) + ".");
        }
        catch (RuntimeException | ArchiveException e) {
            e.printStackTrace();
            this.messageService.logMessage(MESSAGE_TOPIC, "Export failed " + ("0".equals(lastUniqueID) ? " (failed to save metadata)" : "(last uniqueid=" + lastUniqueID + ")") + " : " + e.getMessage());
            this.stop(timer);
        }
    }

    public ItemCollection reset() throws ArchiveException {
        logger.info("Reset Export SyncPoint...");
        ItemCollection metaData = this.dataService.loadMetadata();
        metaData.setItemValue(ITEM_EXPORTPOINT, (Object)0);
        metaData.setItemValue(ITEM_EXPORTCOUNT, (Object)0);
        metaData.setItemValue(ITEM_EXPORTERRORS, (Object)0);
        metaData.setItemValue(ITEM_EXPORTSIZE, (Object)0);
        this.dataService.saveMetadata(metaData);
        return metaData;
    }

    String findLatestSnapshotID(String snapshotID) {
        String latestSnapshot = snapshotID;
        long snapshotTime = this.dataService.getSnapshotTime(snapshotID);
        List _tmpSnapshots = this.dataService.loadSnapshotsByUnqiueID(this.dataService.getUniqueID(snapshotID));
        for (String _tmpSnapshotID : _tmpSnapshots) {
            logger.finest(".......           :" + _tmpSnapshotID);
        }
        for (String _tmpSnapshotID : _tmpSnapshots) {
            long _tmpSnapshotTime = this.dataService.getSnapshotTime(_tmpSnapshotID);
            if (_tmpSnapshotTime < snapshotTime) continue;
            latestSnapshot = _tmpSnapshotID;
        }
        return latestSnapshot;
    }
}

